<?php

namespace App\Console\Commands;

use App\Models\Tp_Transaction;
use Illuminate\Console\Command;

class RunAutoTrading extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'auto:run-trading';  // Corrected command signature

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Run auto trading for all eligible users';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
{
    $now = \Carbon\Carbon::now();
    \Log::info("Auto Trading Started at: " . now());

    $users = \App\Models\User::where('auto_trading_enabled', 1)->get();
    \Log::info("Eligible users count: " . $users->count());

    foreach ($users as $user) {
        // Ensure the 'last_auto_trade_at' is a Carbon instance
        $lastTradeAt = $user->last_auto_trade_at ? \Carbon\Carbon::parse($user->last_auto_trade_at) : null;

        // Default to false, will be updated based on the trading frequency
        $shouldRun = false;

        // Compare against the appropriate time difference based on frequency
        switch ($user->auto_trading_frequency) {
            case 'minute':
                $shouldRun = $lastTradeAt === null || $lastTradeAt->diffInSeconds($now) >= 60;
                break;
            case 'hourly':
                $shouldRun = $lastTradeAt === null || $lastTradeAt->diffInHours($now, false) >= 1 || $lastTradeAt->diffInMinutes($now, false) >= 59;
                break;
            case 'daily':
                $shouldRun = $lastTradeAt === null || $lastTradeAt->diffInDays($now, false) >= 1 || $lastTradeAt->diffInHours($now, false) >= 23;
                break;
            case 'weekly':
                $shouldRun = $lastTradeAt === null || $lastTradeAt->diffInWeeks($now, false) >= 1 || $lastTradeAt->diffInDays($now, false) >= 6;
                break;
            case 'monthly':
                $shouldRun = $lastTradeAt === null || $lastTradeAt->diffInMonths($now, false) >= 1 || $lastTradeAt->diffInDays($now, false) >= 29;
                break;
        }



        if ($shouldRun) {
            // Calculate the profit
            $profit = ($user->account_bal * $user->auto_trading_percent) / 100;

            // Update user account balance, ROI and last auto trade time
            $user->account_bal = $user->account_bal + $profit;
            $user->roi = $user->roi + $profit;
            $user->last_auto_trade_at = $now;
            $user->save();

            // Record the auto trading transaction
            Tp_Transaction::create([
                'user' => $user->id,
                'plan' => "Auto Trading",
                'amount' => $profit,
                'type' => "Profit",
            ]);

            \Log::info("User {$user->id} auto-traded. Profit: $profit");
        } else {
            \Log::info("User {$user->id} skipped. Frequency not met.");
        }
    }

    $this->info('Auto trading processed successfully.');
    return 0;
}

}
