<?php

namespace App\Jobs;

use App\Models\Copy;
use App\Models\Expert;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;  // Import Log Facade

class CalculateCopyTradeProfit implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        Log::info('CalculateCopyTradeProfit job started.');  // Log job start

        // Get all active copied trades
        $copies = Copy::where('status', 'active')->get();

        Log::info('Found ' . $copies->count() . ' active copied trades.');  // Log number of active copies

        foreach ($copies as $copy) {
            $user = User::find($copy->user_id);
            $expert = Expert::find($copy->expert_id);

            if (!$user || !$expert) {
                Log::warning('Missing user or expert for copy ID: ' . $copy->id);  // Log warning if user/expert is missing
                continue; // Skip if user or expert is missing
            }

            // Check if the trade is a WIN based on the expert's win rate
            $winChance = $expert->win_rate; // e.g., 70% win rate
            $isWin = rand(1, 100) <= $winChance;

            Log::info('Copy ID: ' . $copy->id . ' - Trade ' . ($isWin ? 'won' : 'lost') . '.');  // Log win/loss result

            $profit = 0;

            if ($isWin) {
                // Calculate profit based on profit percentage set by admin
                $profit = ($expert->profit_percentage / 100) * $copy->total_profit;
                Log::info('Copy ID: ' . $copy->id . ' - Profit calculated: ' . $profit);  // Log calculated profit
            }

            // Calculate profit share for expert
            $expertShare = ($expert->profit_share / 100) * $profit;
            $userProfit = $profit - $expertShare;

            // Update user's total profit and balance
            $user->account_bal += $userProfit;
            $user->roi += $userProfit;
            $user->save();

            Log::info('User ID: ' . $user->id . ' - Updated balance: ' . $user->account_bal);  // Log updated user balance

            // Update copy trade profit
            $copy->total_profit += $userProfit;
            $copy->save();

            Log::info('Copy ID: ' . $copy->id . ' - Updated total profit: ' . $copy->total_profit);  // Log updated copy profit

            // Update expert's total profit
            $expert->total_profit += $expertShare;
            $expert->save();

            Log::info('Expert ID: ' . $expert->id . ' - Updated expert total profit: ' . $expert->total_profit);  // Log updated expert profit
        }

        Log::info('CalculateCopyTradeProfit job finished.');  // Log job completion
    }
}
