'use strict';

var _should = require('should');

var _should2 = _interopRequireDefault(_should);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _httpClient = require('../httpClient');

var _httpClient2 = _interopRequireDefault(_httpClient);

var _configuration = require('./configuration.client');

var _configuration2 = _interopRequireDefault(_configuration);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const copyFactoryApiUrl = 'https://copyfactory-application-history-master-v1.agiliumtrade.agiliumtrade.ai';

/**
 * @test {ConfigurationClient}
 */
describe('ConfigurationClient', () => {

  let sandbox;
  let copyFactoryClient;
  const token = 'header.payload.sign';
  let httpClient = new _httpClient2.default();
  let requestStub;

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(() => {
    copyFactoryClient = new _configuration2.default(httpClient, token);
    requestStub = sandbox.stub(httpClient, 'request');
  });

  afterEach(() => {
    sandbox.restore();
  });

  /**
   * @test {ConfigurationClient#generateAccountId}
   */
  it('should generate account id', async () => {
    copyFactoryClient.generateAccountId().length.should.equal(64);
  });

  /**
   * @test {ConfigurationClient#generateStrategyId}
   */
  it('should generate strategy id', async () => {
    let expected = {
      id: 'ABCD'
    };
    requestStub.resolves(expected);
    let id = await copyFactoryClient.generateStrategyId();
    id.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/unused-strategy-id`,
      method: 'GET',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {ConfigurationClient#generateStrategyId}
   */
  it('should not generate strategy id with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.generateStrategyId();
    } catch (error) {
      error.message.should.equal('You can not invoke generateStrategyId method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ConfigurationClient#getStrategies}
   */
  it('should retrieve strategies from API', async () => {
    let expected = [{
      _id: 'ABCD',
      platformCommissionRate: 0.01,
      name: 'Test strategy',
      accountId: 'e8867baa-5ec2-45ae-9930-4d5cea18d0d6',
      maxTradeRisk: 0.1,
      stopOutRisk: {
        relativeValue: 0.4,
        startTime: '2020-08-24T00:00:00.000Z'
      },
      riskLimits: [{
        type: 'monthly',
        applyTo: 'balance',
        maxRelativeRisk: 0.5,
        closePositions: false,
        startTime: '2020-08-24T00:00:01.000Z'
      }],
      timeSettings: {
        lifetimeInHours: 192,
        openingIntervalInMinutes: 5
      }
    }];
    requestStub.resolves(expected);
    let strategies = await copyFactoryClient.getStrategies(true, 100, 200);
    strategies.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/strategies`,
      method: 'GET',
      headers: {
        'auth-token': token
      },
      qs: {
        includeRemoved: true,
        limit: 100,
        offset: 200
      },
      json: true
    });
  });

  /**
     * @test {ConfigurationClient#getStrategies}
     */
  it('should not retrieve strategies from API with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.getStrategies();
    } catch (error) {
      error.message.should.equal('You can not invoke getStrategies method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ConfigurationClient#getStrategy}
   */
  it('should retrieve strategy from API', async () => {
    let expected = {
      _id: 'ABCD',
      providerId: 'providerId',
      platformCommissionRate: 0.01,
      name: 'Test strategy',
      connectionId: 'e8867baa-5ec2-45ae-9930-4d5cea18d0d6',
      maxTradeRisk: 0.1,
      stopOutRisk: {
        relativeValue: 0.4,
        startTime: '2020-08-24T00:00:00.000Z'
      },
      timeSettings: {
        lifetimeInHours: 192,
        openingIntervalInMinutes: 5
      }
    };
    requestStub.resolves(expected);
    let strategies = await copyFactoryClient.getStrategy('ABCD');
    strategies.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/strategies/ABCD`,
      method: 'GET',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
     * @test {ConfigurationClient#getStrategy}
     */
  it('should not retrieve strategy from API with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.getStrategy('ABCD');
    } catch (error) {
      error.message.should.equal('You can not invoke getStrategy method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ConfigurationClient#updateStrategy}
   */
  it('should update strategy via API', async () => {
    const strategy = {
      name: 'Test strategy',
      accountId: 'e8867baa-5ec2-45ae-9930-4d5cea18d0d6',
      maxTradeRisk: 0.1,
      stopOutRisk: {
        relativeValue: 0.4,
        startTime: '2020-08-24T00:00:00.000Z'
      },
      riskLimits: [{
        type: 'monthly',
        applyTo: 'balance',
        maxRelativeRisk: 0.5,
        closePositions: false,
        startTime: '2020-08-24T00:00:01.000Z'
      }],
      timeSettings: {
        lifetimeInHours: 192,
        openingIntervalInMinutes: 5
      }
    };
    await copyFactoryClient.updateStrategy('ABCD', strategy);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/strategies/ABCD`,
      method: 'PUT',
      headers: {
        'auth-token': token
      },
      json: true,
      body: strategy });
  });

  /**
   * @test {ConfigurationClient#updateStrategy}
   */
  it('should not update strategy via API with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.updateStrategy('ABCD', {});
    } catch (error) {
      error.message.should.equal('You can not invoke updateStrategy method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ConfigurationClient#removeStrategy}
   */
  it('should remove strategy via API', async () => {
    const payload = { mode: 'preserve', removeAfter: '2020-08-24T00:00:00.000Z' };
    await copyFactoryClient.removeStrategy('ABCD', payload);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/strategies/ABCD`,
      method: 'DELETE',
      headers: {
        'auth-token': token
      },
      body: payload,
      json: true
    });
  });

  /**
   * @test {ConfigurationClient#removeStrategy}
   */
  it('should not remove strategy from via with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.removeStrategy('ABCD');
    } catch (error) {
      error.message.should.equal('You can not invoke removeStrategy method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ConfigurationClient#getPortfolioStrategies}
   */
  it('should retrieve portfolio strategies from API', async () => {
    let expected = [{
      _id: 'ABCD',
      providerId: 'providerId',
      platformCommissionRate: 0.01,
      name: 'Test strategy',
      members: [{
        strategyId: 'BCDE'
      }],
      maxTradeRisk: 0.1,
      stopOutRisk: {
        relativeValue: 0.4,
        startTime: '2020-08-24T00:00:00.000Z'
      }
    }];
    requestStub.resolves(expected);
    let strategies = await copyFactoryClient.getPortfolioStrategies(true, 100, 200);
    strategies.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/portfolio-strategies`,
      method: 'GET',
      headers: {
        'auth-token': token
      },
      qs: {
        includeRemoved: true,
        limit: 100,
        offset: 200
      },
      json: true
    });
  });

  /**
   * @test {ConfigurationClient#getPortfolioStrategies}
   */
  it('should not retrieve portfolio strategies from API with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.getPortfolioStrategies();
    } catch (error) {
      error.message.should.equal('You can not invoke getPortfolioStrategies method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ConfigurationClient#getPortfolioStrategy}
   */
  it('should retrieve portfolio strategy from API', async () => {
    let expected = {
      _id: 'ABCD',
      providerId: 'providerId',
      platformCommissionRate: 0.01,
      name: 'Test strategy',
      members: [{
        strategyId: 'BCDE'
      }],
      maxTradeRisk: 0.1,
      stopOutRisk: {
        relativeValue: 0.4,
        startTime: '2020-08-24T00:00:00.000Z'
      }
    };
    requestStub.resolves(expected);
    let strategies = await copyFactoryClient.getPortfolioStrategy('ABCD');
    strategies.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/portfolio-strategies/ABCD`,
      method: 'GET',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {ConfigurationClient#getPortfolioStrategy}
   */
  it('should not retrieve portfolio strategy from API with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.getPortfolioStrategy('ABCD');
    } catch (error) {
      error.message.should.equal('You can not invoke getPortfolioStrategy method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ConfigurationClient#updatePortfolioStrategy}
   */
  it('should update portfolio strategy via API', async () => {
    const strategy = {
      name: 'Test strategy',
      members: [{
        strategyId: 'BCDE'
      }],
      maxTradeRisk: 0.1,
      stopOutRisk: {
        relativeValue: 0.4,
        startTime: '2020-08-24T00:00:00.000Z'
      }
    };
    await copyFactoryClient.updatePortfolioStrategy('ABCD', strategy);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/portfolio-strategies/ABCD`,
      method: 'PUT',
      headers: {
        'auth-token': token
      },
      json: true,
      body: strategy
    });
  });

  /**
   * @test {ConfigurationClient#updatePortfolioStrategy}
   */
  it('should not update portfolio strategy via API with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.updatePortfolioStrategy('ABCD', {});
    } catch (error) {
      error.message.should.equal('You can not invoke updatePortfolioStrategy method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ConfigurationClient#removePortfolioStrategy}
   */
  it('should remove portfolio strategy via API', async () => {
    const payload = { mode: 'preserve', removeAfter: '2020-08-24T00:00:00.000Z' };
    await copyFactoryClient.removePortfolioStrategy('ABCD', payload);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/portfolio-strategies/ABCD`,
      method: 'DELETE',
      headers: {
        'auth-token': token
      },
      body: payload,
      json: true
    });
  });

  /**
   * @test {ConfigurationClient#removePortfolioStrategy}
   */
  it('should not remove portfolio strategy from via with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.removePortfolioStrategy('ABCD');
    } catch (error) {
      error.message.should.equal('You can not invoke removePortfolioStrategy method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {ConfigurationClient#removePortfolioStrategyMember}
   */
  it('should remove portfolio strategy member via API', async () => {
    const payload = { mode: 'preserve', removeAfter: '2020-08-24T00:00:00.000Z' };
    await copyFactoryClient.removePortfolioStrategyMember('ABCD', 'BCDE', payload);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/portfolio-strategies/ABCD/members/BCDE`,
      method: 'DELETE',
      headers: {
        'auth-token': token
      },
      body: payload,
      json: true
    });
  });

  /**
     * @test {ConfigurationClient#removePortfolioStrategyMember}
     */
  it('should not remove portfolio strategy member from via with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.removePortfolioStrategyMember('ABCD', 'BCDE');
    } catch (error) {
      error.message.should.equal('You can not invoke removePortfolioStrategyMember method, because you have connected with account access ' + 'token. Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {TradingClient#getSubscribers}
   */
  it('should retrieve CopyFactory subscribers from API', async () => {
    let expected = [{
      _id: 'e8867baa-5ec2-45ae-9930-4d5cea18d0d6',
      name: 'Demo account',
      reservedMarginFraction: 0.25,
      subscriptions: [{
        strategyId: 'ABCD',
        multiplier: 1
      }]
    }];
    requestStub.resolves(expected);
    let accounts = await copyFactoryClient.getSubscribers(true, 100, 200);
    accounts.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/subscribers`,
      method: 'GET',
      headers: {
        'auth-token': token
      },
      qs: {
        includeRemoved: true,
        limit: 100,
        offset: 200
      },
      json: true
    });
  });

  /**
     * @test {TradingClient#getSubscribers}
     */
  it('should not retrieve subscribers from API with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.getSubscribers();
    } catch (error) {
      error.message.should.equal('You can not invoke getSubscribers method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {TradingClient#getSubscriber}
   */
  it('should retrieve CopyFactory subscriber from API', async () => {
    let expected = {
      _id: 'e8867baa-5ec2-45ae-9930-4d5cea18d0d6',
      name: 'Demo account',
      reservedMarginFraction: 0.25,
      subscriptions: [{
        strategyId: 'ABCD',
        multiplier: 1
      }]
    };
    requestStub.resolves(expected);
    let accounts = await copyFactoryClient.getSubscriber('e8867baa-5ec2-45ae-9930-4d5cea18d0d6');
    accounts.should.equal(expected);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/subscribers/` + 'e8867baa-5ec2-45ae-9930-4d5cea18d0d6',
      method: 'GET',
      headers: {
        'auth-token': token
      },
      json: true
    });
  });

  /**
   * @test {TradingClient#getSubscriber}
   */
  it('should not retrieve CopyFactory subscriber from API with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.getSubscriber('test');
    } catch (error) {
      error.message.should.equal('You can not invoke getSubscriber method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {TradingClient#updateSubscriber}
   */
  it('should update CopyFactory subscriber via API', async () => {
    let subscriber = {
      name: 'Demo account',
      reservedMarginFraction: 0.25,
      subscriptions: [{
        strategyId: 'ABCD',
        multiplier: 1
      }]
    };
    await copyFactoryClient.updateSubscriber('e8867baa-5ec2-45ae-9930-4d5cea18d0d6', subscriber);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/subscribers/` + 'e8867baa-5ec2-45ae-9930-4d5cea18d0d6',
      method: 'PUT',
      headers: {
        'auth-token': token
      },
      body: subscriber,
      json: true
    });
  });

  /**
   * @test {TradingClient#updateSubscriber}
   */
  it('should not update CopyFactory subscriber via API with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.updateSubscriber('test', {});
    } catch (error) {
      error.message.should.equal('You can not invoke updateSubscriber method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {TradingClient#removeSubscriber}
   */
  it('should remove CopyFactory subscriber via API', async () => {
    const payload = { mode: 'preserve', removeAfter: '2020-08-24T00:00:00.000Z' };
    await copyFactoryClient.removeSubscriber('e8867baa-5ec2-45ae-9930-4d5cea18d0d6', payload);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/subscribers/` + 'e8867baa-5ec2-45ae-9930-4d5cea18d0d6',
      method: 'DELETE',
      headers: {
        'auth-token': token
      },
      body: payload,
      json: true
    });
  });

  /**
   * @test {TradingClient#removeSubscriber}
   */
  it('should not remove CopyFactory subscriber via API with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.removeSubscriber('test');
    } catch (error) {
      error.message.should.equal('You can not invoke removeSubscriber method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });

  /**
   * @test {TradingClient#removeSubscription}
   */
  it('should remove CopyFactory subscription via API', async () => {
    const payload = { mode: 'preserve', removeAfter: '2020-08-24T00:00:00.000Z' };
    await copyFactoryClient.removeSubscription('e8867baa-5ec2-45ae-9930-4d5cea18d0d6', 'ABCD', payload);
    _sinon2.default.assert.calledOnceWithExactly(httpClient.request, {
      url: `${copyFactoryApiUrl}/users/current/configuration/subscribers/` + 'e8867baa-5ec2-45ae-9930-4d5cea18d0d6/subscriptions/ABCD',
      method: 'DELETE',
      headers: {
        'auth-token': token
      },
      body: payload,
      json: true
    });
  });

  /**
   * @test {TradingClient#removeSubscription}
   */
  it('should not remove CopyFactory subscription via API with account token', async () => {
    copyFactoryClient = new _configuration2.default(httpClient, 'token');
    try {
      await copyFactoryClient.removeSubscription('test', 'ABCD');
    } catch (error) {
      error.message.should.equal('You can not invoke removeSubscription method, because you have connected with account access token. ' + 'Please use API access token from https://app.metaapi.cloud/token page to invoke this method.');
    }
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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