'use strict';

var _should = require('should');

var _should2 = _interopRequireDefault(_should);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _httpClient = require('./httpClient');

var _httpClient2 = _interopRequireDefault(_httpClient);

var _errorHandler = require('./errorHandler');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @test {HttpClient#request}
 */
describe('HttpClient#request', () => {

  let httpClient;

  /**
   * @test {HttpClient#request}
   */
  describe('Real request', () => {

    beforeEach(() => {
      httpClient = new _httpClient2.default();
    });

    /**
     * @test {HttpClient#request}
     */
    it('should load HTML page from example.com', async () => {
      const opts = { url: 'http://example.com' };
      const response = await httpClient.request(opts);
      response.should.match(/doctype html/);
    });

    /**
     * @test {HttpClient#request}
     */
    it('should return NotFound error if server returns 404', async () => {
      let opts = { url: 'http://example.com/not-found' };
      try {
        const response = await httpClient.request(opts);
        _should2.default.not.exist(response);
      } catch (err) {
        err.name.should.be.eql('NotFoundError');
      }
    });

    /**
     * @test {HttpClient#request}
     */
    it('should return timeout error if request is timed out', async () => {
      httpClient = new _httpClient2.default(0.001, { retries: 2 });
      let opts = { url: 'http://metaapi.cloud' };
      try {
        const response = await httpClient.request(opts);
        _should2.default.not.exist(response);
      } catch (err) {
        err.name.should.be.eql('ApiError');
        err.message.should.be.eql('ETIMEDOUT');
      }
    }).timeout(10000);
  });

  /**
   * @test {HttpClient#request}
   */
  describe('Retry request', () => {

    const opts = { url: 'http://metaapi.cloud' };
    let sandbox, stub;

    before(() => {
      sandbox = _sinon2.default.createSandbox();
    });

    beforeEach(() => {
      stub = sandbox.stub();
      httpClient = new _httpClient.HttpClientMock(stub);
    });

    afterEach(() => {
      sandbox.restore();
    });

    /**
     * @test {HttpClient#request}
     */
    describe('when InternalError or ApiError error occured', () => {

      /**
       * @test {HttpClient#request}
       */
      it('should retry request on fail with ApiError error', async () => {
        stub.onFirstCall().rejects(new _errorHandler.ApiError(_errorHandler.ApiError, 'test')).onSecondCall().rejects(new _errorHandler.ApiError(_errorHandler.ApiError, 'test')).onThirdCall().resolves('response');
        const response = await httpClient.request(opts);
        response.should.match('response');
        _sinon2.default.assert.calledThrice(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should retry request on fail with InternalError error', async () => {
        stub.onFirstCall().rejects(new _errorHandler.InternalError('test')).onSecondCall().rejects(new _errorHandler.InternalError('test')).onThirdCall().resolves('response');
        const response = await httpClient.request(opts);
        response.should.match('response');
        _sinon2.default.assert.calledThrice(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should return error if retry limit exceeded', async () => {
        stub.rejects(new _errorHandler.ApiError(_errorHandler.ApiError, 'test'));
        httpClient = new _httpClient.HttpClientMock(stub, 60, { retries: 2 });
        try {
          const response = await httpClient.request(opts);
          _should2.default.not.exist(response);
        } catch (err) {
          err.name.should.eql('ApiError');
          err.message.should.eql('test');
        }
        _sinon2.default.assert.calledThrice(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should not retry if error is neither InternalError nor ApiError', async () => {
        stub.onFirstCall().rejects(new _errorHandler.ValidationError('test')).onSecondCall().rejects(new _errorHandler.ValidationError('test')).onThirdCall().resolves('response');
        try {
          const response = await httpClient.request(opts);
          _should2.default.not.exist(response);
        } catch (err) {
          err.name.should.eql('ValidationError');
          err.message.should.eql('test');
        }
        _sinon2.default.assert.calledOnce(stub);
      }).timeout(10000);
    });

    /**
     * @test {HttpClient#request}
     */
    describe('when TooManyRequestsError error occured', () => {

      const getTooManyRequestsError = sec => {
        const date = new Date();
        date.setSeconds(date.getSeconds() + sec);
        const recommendedRetryTime = date.toUTCString();
        return new _errorHandler.TooManyRequestsError('test', { recommendedRetryTime });
      };

      /**
       * @test {HttpClient#request}
       */
      it('should retry request after waiting on fail with TooManyRequestsError error', async () => {
        stub.onFirstCall().rejects(getTooManyRequestsError(2)).onSecondCall().rejects(getTooManyRequestsError(3)).onThirdCall().resolves('response');
        const response = await httpClient.request(opts);
        response.should.eql('response');
        _sinon2.default.assert.calledThrice(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should return error if recommended retry time is too long', async () => {
        stub.onFirstCall().rejects(getTooManyRequestsError(2)).onSecondCall().rejects(getTooManyRequestsError(300)).onThirdCall().resolves('response');
        try {
          const response = await httpClient.request(opts);
          _should2.default.not.exist(response);
        } catch (err) {
          err.name.should.eql('TooManyRequestsError');
          err.message.should.eql('test');
        }
        _sinon2.default.assert.calledTwice(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should not count retrying TooManyRequestsError error', async () => {
        stub.onFirstCall().rejects(getTooManyRequestsError(1)).onSecondCall().rejects(new _errorHandler.ApiError(_errorHandler.ApiError, 'test')).onThirdCall().resolves('response');
        httpClient = new _httpClient.HttpClientMock(stub, 60, { retries: 1 });
        const response = await httpClient.request(opts);
        response.should.eql('response');
        _sinon2.default.assert.calledThrice(stub);
      }).timeout(10000);
    });

    /**
     * @test {HttpClient#request}
     */
    describe('when status 202 response received', () => {

      /**
       * @test {HttpClient#request}
       */
      it('should wait for the retry-after header time before retrying', async () => {
        stub.callsFake(options => {
          options.callback(null, { headers: { 'retry-after': 3 }, statusCode: 202 });
        }).onThirdCall().resolves('response');
        const response = await httpClient.request(opts);
        response.should.eql('response');
        _sinon2.default.assert.calledThrice(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should return TimeoutError error if retry-after header time is too long', async () => {
        stub.callsFake(options => {
          options.callback(null, { headers: { 'retry-after': 30 }, statusCode: 202 });
        });
        httpClient = new _httpClient.HttpClientMock(stub, 60, { maxDelayInSeconds: 3 });
        try {
          await httpClient.request(opts);
          _should2.default.not.exist('Should not exist this assertion');
        } catch (err) {
          err.name.should.eql('TimeoutError');
          err.message.should.eql('Timed out waiting for the end of the process of calculating metrics');
        }
        _sinon2.default.assert.calledOnce(stub);
      }).timeout(10000);

      /**
       * @test {HttpClient#request}
       */
      it('should return TimeoutError error if timed out to retry', async () => {
        stub.callsFake(options => {
          options.callback(null, { headers: { 'retry-after': 1 }, statusCode: 202 });
        });
        httpClient = new _httpClient.HttpClientMock(stub, 60, { maxDelayInSeconds: 2, retries: 3 });
        try {
          await httpClient.request(opts);
          _should2.default.not.exist('Should not exist this assertion');
        } catch (err) {
          err.name.should.eql('TimeoutError');
          err.message.should.eql('Timed out waiting for the end of the process of calculating metrics');
        }
        _sinon2.default.assert.callCount(stub, 6);
      }).timeout(10000);
    });
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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