'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _fs = require('fs');

var _fs2 = _interopRequireDefault(_fs);

var _metaApi = require('../metaApi.client');

var _metaApi2 = _interopRequireDefault(_metaApi);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * metaapi.cloud provisioning profile API client (see https://metaapi.cloud/docs/provisioning/)
 */
class ProvisioningProfileClient extends _metaApi2.default {

  /**
   * Provisioning profile model
   * @typedef {Object} ProvisioningProfileDto
   * @property {String} _id provisioning profile unique identifier
   * @property {String} name provisioning profile name
   * @property {Number} version MetaTrader version (allowed values are 4 and 5)
   * @property {String} status provisioning profile status (allowed values are new and active)
   * @property {String} brokerTimezone broker timezone name from Time Zone Database
   * @property {String} brokerDSTSwitchTimezone broker DST switch timezone name from Time Zone Database
   */

  /**
   * Retrieves provisioning profiles owned by user
   * (see https://metaapi.cloud/docs/provisioning/api/provisioningProfile/readProvisioningProfiles/)
   * Method is accessible only with API access token
   * @param {Number} version optional version filter (allowed values are 4 and 5)
   * @param {String} status optional status filter (allowed values are new and active)
   * @return {Promise<Array<ProvisioningProfileDto>>} promise resolving with provisioning profiles found
   */
  getProvisioningProfiles(version, status) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getProvisioningProfiles');
    }
    let qs = {};
    if (version) {
      qs.version = version;
    }
    if (status) {
      qs.status = status;
    }
    const opts = {
      url: `${this._host}/users/current/provisioning-profiles`,
      method: 'GET',
      qs,
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Retrieves a provisioning profile by id (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/readProvisioningProfile/). Throws an error if
   * profile is not found.
   * Method is accessible only with API access token
   * @param {String} id provisioning profile id
   * @return {Promise<ProvisioningProfileDto>} promise resolving with provisioning profile found
   */
  getProvisioningProfile(id) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getProvisioningProfile');
    }
    const opts = {
      url: `${this._host}/users/current/provisioning-profiles/${id}`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * New provisioning profile model
   * @typedef {Object} NewProvisioningProfileDto
   * @property {String} name provisioning profile name
   * @property {Number} version MetaTrader version (allowed values are 4 and 5)
   * @property {String} brokerTimezone broker timezone name from Time Zone Database
   * @property {String} brokerDSTSwitchTimezone broker DST switch timezone name from Time Zone Database
   */

  /**
   * Provisioning profile id model
   * @typedef {Object} ProvisioningProfileIdDto
   * @property {String} id provisioning profile unique identifier
   */

  /**
   * Creates a new provisioning profile (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/createNewProvisioningProfile/). After creating a
   * provisioning profile you are required to upload extra files in order to activate the profile for further use.
   * Method is accessible only with API access token
   * @param {NewProvisioningProfileDto} provisioningProfile provisioning profile to create
   * @return {Promise<ProvisioningProfileIdDto>} promise resolving with an id of the provisioning profile created
   */
  createProvisioningProfile(provisioningProfile) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('createProvisioningProfile');
    }
    const opts = {
      url: `${this._host}/users/current/provisioning-profiles`,
      method: 'POST',
      headers: {
        'auth-token': this._token
      },
      json: true,
      body: provisioningProfile
    };
    return this._httpClient.request(opts);
  }

  /**
   * Uploads a file to a provisioning profile (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/uploadFilesToProvisioningProfile/). Uploading a
   * file by name is allowed only for Node.js.
   * Method is accessible only with API access token
   * @param {String} provisioningProfileId provisioning profile id to upload file to
   * @param {String} fileName name of the file to upload. Allowed values are servers.dat for MT5 profile, broker.srv for
   * MT4 profile
   * @param {String|Buffer} file path to a file to upload or buffer containing file contents
   * @return {Promise} promise resolving when file upload is completed
   */
  uploadProvisioningProfileFile(provisioningProfileId, fileName, file) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('uploadProvisioningProfileFile');
    }
    if (typeof file === 'string') {
      file = _fs2.default.createReadStream(file);
    } else {
      file = {
        value: file,
        options: {
          filename: 'serverFile'
        }
      };
    }
    const opts = {
      method: 'PUT',
      url: `${this._host}/users/current/provisioning-profiles/${provisioningProfileId}/${fileName}`,
      formData: {
        file
      },
      json: true,
      headers: {
        'auth-token': this._token
      }
    };
    return this._httpClient.request(opts);
  }

  /**
   * Deletes a provisioning profile (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/deleteProvisioningProfile/). Please note that in
   * order to delete a provisioning profile you need to delete MT accounts connected to it first.
   * Method is accessible only with API access token
   * @param {String} id provisioning profile id
   * @return {Promise} promise resolving when provisioning profile is deleted
   */
  deleteProvisioningProfile(id) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('deleteProvisioningProfile');
    }
    const opts = {
      url: `${this._host}/users/current/provisioning-profiles/${id}`,
      method: 'DELETE',
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    return this._httpClient.request(opts);
  }

  /**
   * Updated provisioning profile data
   * @typedef {Object} ProvisioningProfileUpdateDto
   * @property {String} name provisioning profile name
   */

  /**
   * Updates existing provisioning profile data (see
   * https://metaapi.cloud/docs/provisioning/api/provisioningProfile/updateProvisioningProfile/).
   * Method is accessible only with API access token
   * @param {String} id provisioning profile id
   * @param {ProvisioningProfileUpdateDto} provisioningProfile updated provisioning profile
   * @return {Promise} promise resolving when provisioning profile is updated
   */
  updateProvisioningProfile(id, provisioningProfile) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('updateProvisioningProfile');
    }
    const opts = {
      url: `${this._host}/users/current/provisioning-profiles/${id}`,
      method: 'PUT',
      headers: {
        'auth-token': this._token
      },
      json: true,
      body: provisioningProfile
    };
    return this._httpClient.request(opts);
  }

}
exports.default = ProvisioningProfileClient;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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