'use strict';

/**
 * Implements an expert advisor entity
 */

Object.defineProperty(exports, "__esModule", {
  value: true
});
class ExpertAdvisor {

  /**
   * Constructs an expert advisor entity
   * @param {ExpertAdvisorDto} data
   * @param accountId
   * @param expertAdvisorClient
   */
  constructor(data, accountId, expertAdvisorClient) {
    this._data = data;
    this._accountId = accountId;
    this._expertAdvisorClient = expertAdvisorClient;
  }

  /**
   * Returns expert id
   * @returns {String} expert id
   */
  get expertId() {
    return this._data.expertId;
  }

  /**
   * Returns expert period
   * @returns {String} expert period
   */
  get period() {
    return this._data.period;
  }

  /**
   * Returns expert symbol
   * @returns {String} expert symbol
   */
  get symbol() {
    return this._data.symbol;
  }

  /**
   * Returns true if expert file was uploaded
   * @returns {Boolean}
   */
  get fileUploaded() {
    return this._data.fileUploaded;
  }

  /**
   * Reloads expert advisor from API
   * (see https://metaapi.cloud/docs/provisioning/api/expertAdvisor/readExpertAdvisor/)
   * @returns {Promise} promise resolving when expert advisor is updated
   */
  async reload() {
    this._data = await this._expertAdvisorClient.getExpertAdvisor(this._accountId, this.expertId);
  }

  /**
   * Updates expert advisor data
   * (see https://metaapi.cloud/docs/provisioning/api/expertAdvisor/updateExpertAdvisor/)
   * @param {NewExpertAdvisorDto} expert new expert advisor data
   * @returns {Promise} promise resolving when expert advisor is updated
   */
  async update(expert) {
    await this._expertAdvisorClient.updateExpertAdvisor(this._accountId, this.expertId, expert);
    await this.reload();
  }

  /**
   * Uploads an expert advisor file. EAs which use DLLs are not supported
   * (see https://metaapi.cloud/docs/provisioning/api/expertAdvisor/uploadEAFile/)
   * @param {String|Buffer} file expert advisor file
   * @returns {Promise} promise resolving when file upload is completed
   */
  async uploadFile(file) {
    await this._expertAdvisorClient.uploadExpertAdvisorFile(this._accountId, this.expertId, file);
    await this.reload();
  }

  /**
   * Removes expert advisor
   * (see https://metaapi.cloud/docs/provisioning/api/expertAdvisor/deleteExpertAdvisor/)
   * @returns {Promise} promise resolving when expert advisor removed
   */
  async remove() {
    await this._expertAdvisorClient.deleteExpertAdvisor(this._accountId, this.expertId);
  }

}
exports.default = ExpertAdvisor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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