'use strict';

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _browserManager = require('./browserManager');

var _browserManager2 = _interopRequireDefault(_browserManager);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

require('fake-indexeddb/auto');
const { openDB, deleteDB } = require('idb');

/**
 * Helper function to get the test database
 * @returns {IndexedDB} Indexed DB
 */
async function getTestDb() {
  const db = await openDB('metaapi', 1, {
    upgrade(database) {
      if (!database.objectStoreNames.contains('deals')) {
        database.createObjectStore('deals', { keyPath: 'accountIdAndApplication' });
      }
      if (!database.objectStoreNames.contains('historyOrders')) {
        database.createObjectStore('historyOrders', { keyPath: 'accountIdAndApplication' });
      }
    }
  });
  return db;
}

/**
 * Helper function to read saved history storage
 * @returns {history} history object
 */
async function readHistoryStorageFile() {
  const history = { deals: [], historyOrders: [] };
  const db = await getTestDb();
  history.deals = await db.get('deals', 'accountId-application');
  history.historyOrders = await db.get('historyOrders', 'accountId-application');
  db.close();
  return history;
}

/**
 * Helper function to create test data in IndexedDB
 * @param deals history deals
 * @param historyOrders history orders
 */
async function createTestData(deals, historyOrders, config) {
  const db = await getTestDb();
  await db.put('deals', {
    accountIdAndApplication: 'accountId-application',
    items: deals,
    lastDealTimeByInstanceIndex: config.lastDealTimeByInstanceIndex
  });
  await db.put('historyOrders', {
    accountIdAndApplication: 'accountId-application',
    items: historyOrders,
    lastHistoryOrderTimeByInstanceIndex: config.lastHistoryOrderTimeByInstanceIndex
  });
  db.close();
}

/**
 * @test {BrowserHistoryManager}
 */
describe('BrowserHistoryManger', () => {

  let storage;
  let fileManager;
  let testDeal;
  let testDeal2;
  let testOrder;
  let testOrder2;
  let testConfig;
  let sandbox;

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(async () => {
    storage = {};
    fileManager = new _browserManager2.default('accountId', 'application', storage);
    sandbox.stub(fileManager, 'startUpdateJob').returns();
    testDeal = { id: '37863643', type: 'DEAL_TYPE_BALANCE', magic: 0, time: new Date(100), commission: 0,
      swap: 0, profit: 10000, platform: 'mt5', comment: 'Demo deposit 1' };
    testDeal2 = { id: '37863644', type: 'DEAL_TYPE_SELL', magic: 1, time: new Date(200), commission: 0,
      swap: 0, profit: 10000, platform: 'mt5', comment: 'Demo deposit 2' };
    testOrder = { id: '61210463', type: 'ORDER_TYPE_SELL', state: 'ORDER_STATE_FILLED', symbol: 'AUDNZD', magic: 0,
      time: new Date(50), doneTime: new Date(100), currentPrice: 1, volume: 0.01,
      currentVolume: 0, positionId: '61206630', platform: 'mt5', comment: 'AS_AUDNZD_5YyM6KS7Fv:' };
    testOrder2 = { id: '61210464', type: 'ORDER_TYPE_BUY_LIMIT', state: 'ORDER_STATE_FILLED', symbol: 'AUDNZD', magic: 1,
      time: new Date(75), doneTime: new Date(200), currentPrice: 1, volume: 0.01,
      currentVolume: 0, positionId: '61206631', platform: 'mt5', comment: 'AS_AUDNZD_5YyM6KS7Fv:' };
    testConfig = {
      lastDealTimeByInstanceIndex: { '0': 1000000000000 },
      lastHistoryOrderTimeByInstanceIndex: { '0': 1000000000010 }
    };
  });

  afterEach(async () => {
    const db = await getTestDb();
    await db.clear('deals');
    await db.clear('historyOrders');
    db.close();
  });

  /**
   * @test {BrowserHistoryManager#startUpdateJob}
   * @test {BrowserHistoryManager#stopUpdateJob}
   */
  it('should start and stop job', async () => {
    sandbox.restore();
    const clock = _sinon2.default.useFakeTimers();
    sandbox.stub(fileManager, 'updateDiskStorage').returns();
    fileManager.startUpdateJob();
    await clock.tickAsync(61000);
    _sinon2.default.assert.calledOnce(fileManager.updateDiskStorage);
    await clock.tickAsync(61000);
    _sinon2.default.assert.calledTwice(fileManager.updateDiskStorage);
    fileManager.stopUpdateJob();
    await clock.tickAsync(61000);
    _sinon2.default.assert.calledTwice(fileManager.updateDiskStorage);
    fileManager.startUpdateJob();
    await clock.tickAsync(61000);
    _sinon2.default.assert.calledThrice(fileManager.updateDiskStorage);
    fileManager.stopUpdateJob();
    clock.restore();
  });

  /**
   * @test {BrowserHistoryManager#getHistoryFromDisk}
   */
  describe('getHistoryFromDisk', () => {

    /**
     * @test {BrowserHistoryManager#getHistoryFromDisk}
     */
    it('should read history from file', async () => {
      await createTestData([testDeal], [testOrder], testConfig);
      const history = await fileManager.getHistoryFromDisk();
      history.deals.should.match([testDeal]);
      _sinon2.default.assert.match(history.lastDealTimeByInstanceIndex, testConfig.lastDealTimeByInstanceIndex);
      history.historyOrders.should.match([testOrder]);
      _sinon2.default.assert.match(history.lastHistoryOrderTimeByInstanceIndex, testConfig.lastHistoryOrderTimeByInstanceIndex);
    });

    /**
     * @test {BrowserHistoryManager#getHistoryFromDisk}
     */
    it('should return empty history if entries dont exist', async () => {
      const history = await fileManager.getHistoryFromDisk();
      history.deals.should.match([]);
      history.historyOrders.should.match([]);
    });

    /**
     * @test {BrowserHistoryManager#getHistoryFromDisk}
     */
    it('should return empty history if database does not exist', async () => {
      await deleteDB('metaapi');
      const history = await fileManager.getHistoryFromDisk();
      history.deals.should.match([]);
      history.historyOrders.should.match([]);
    });
  });

  /**
   * @test {BrowserHistoryManager#updateDiskStorage}
   */
  describe('updateDiskStorage', () => {

    /**
     * @test {BrowserHistoryManager#updateDiskStorage}
     */
    it('should create storage if doesnt exist', async () => {
      await deleteDB('metaapi');
      storage.deals = [testDeal];
      storage.lastDealTimeByInstanceIndex = testConfig.lastDealTimeByInstanceIndex;
      storage.historyOrders = [testOrder];
      storage.lastHistoryOrderTimeByInstanceIndex = testConfig.lastHistoryOrderTimeByInstanceIndex;
      await fileManager.updateDiskStorage();
      let history = await readHistoryStorageFile();
      _sinon2.default.assert.match(history.deals.items, [testDeal]);
      _sinon2.default.assert.match(history.deals.lastDealTimeByInstanceIndex, testConfig.lastDealTimeByInstanceIndex);
      _sinon2.default.assert.match(history.historyOrders.items, [testOrder]);
      _sinon2.default.assert.match(history.historyOrders.lastHistoryOrderTimeByInstanceIndex, testConfig.lastHistoryOrderTimeByInstanceIndex);
    });

    /**
     * @test {BrowserHistoryManager#updateDiskStorage}
     */
    it('should add data if doesnt exist', async () => {
      let history = await readHistoryStorageFile();
      _sinon2.default.assert.match(history.deals, undefined);
      _sinon2.default.assert.match(history.historyOrders, undefined);
      storage.deals = [testDeal];
      storage.lastDealTimeByInstanceIndex = testConfig.lastDealTimeByInstanceIndex;
      storage.historyOrders = [testOrder];
      storage.lastHistoryOrderTimeByInstanceIndex = testConfig.lastHistoryOrderTimeByInstanceIndex;
      await fileManager.updateDiskStorage();
      history = await readHistoryStorageFile();
      _sinon2.default.assert.match(history.deals.items, [testDeal]);
      _sinon2.default.assert.match(history.deals.lastDealTimeByInstanceIndex, testConfig.lastDealTimeByInstanceIndex);
      _sinon2.default.assert.match(history.historyOrders.items, [testOrder]);
      _sinon2.default.assert.match(history.historyOrders.lastHistoryOrderTimeByInstanceIndex, testConfig.lastHistoryOrderTimeByInstanceIndex);
    });

    /**
     * @test {BrowserHistoryManager#updateDiskStorage}
     */
    it('should update storage', async () => {
      await createTestData([testDeal], [testOrder], testConfig);
      let history = await readHistoryStorageFile();
      _sinon2.default.assert.match(history.deals.items, [testDeal]);
      _sinon2.default.assert.match(history.historyOrders.items, [testOrder]);
      storage.deals = [testDeal, testDeal2];
      storage.historyOrders = [testOrder, testOrder2];
      storage.lastDealTimeByInstanceIndex = { '1': 20000000 };
      storage.lastHistoryOrderTimeByInstanceIndex = { '1': 30000000 };
      await fileManager.updateDiskStorage();
      history = await readHistoryStorageFile();
      _sinon2.default.assert.match(history.deals.items, [testDeal, testDeal2]);
      _sinon2.default.assert.match(history.deals.lastDealTimeByInstanceIndex, { '1': 20000000 });
      _sinon2.default.assert.match(history.historyOrders.items, [testOrder, testOrder2]);
      _sinon2.default.assert.match(history.historyOrders.lastHistoryOrderTimeByInstanceIndex, { '1': 30000000 });
    });

    /**
     * @test {BrowserHistoryManager#updateDiskStorage}
     */
    it('should remake the storage if storage has invalid structure', async () => {
      await deleteDB('metaapi');
      const db = await openDB('metaapi', 1, {
        upgrade(database, oldVersion, newVersion, transaction) {
          database.createObjectStore('wrong', { keyPath: 'accountIdAndApplication' });
        }
      });
      db.close();
      storage.deals = [testDeal];
      storage.lastDealTimeByInstanceIndex = testConfig.lastDealTimeByInstanceIndex;
      storage.historyOrders = [testOrder];
      storage.lastHistoryOrderTimeByInstanceIndex = testConfig.lastHistoryOrderTimeByInstanceIndex;
      await fileManager.updateDiskStorage();
      let history = await readHistoryStorageFile();
      _sinon2.default.assert.match(history.deals.items, [testDeal]);
      _sinon2.default.assert.match(history.deals.lastDealTimeByInstanceIndex, testConfig.lastDealTimeByInstanceIndex);
      _sinon2.default.assert.match(history.historyOrders.items, [testOrder]);
      _sinon2.default.assert.match(history.historyOrders.lastHistoryOrderTimeByInstanceIndex, testConfig.lastHistoryOrderTimeByInstanceIndex);
    });

    /**
     * @test {BrowserHistoryManager#updateDiskStorage}
     */
    it('should remake the storage if object store has invalid structure', async () => {
      await deleteDB('metaapi');
      const db = await openDB('metaapi', 1, {
        upgrade(database, oldVersion, newVersion, transaction) {
          database.createObjectStore('deals', { keyPath: 'wrong' });
          database.createObjectStore('historyOrders', { keyPath: 'wrong' });
        }
      });
      db.close();
      storage.deals = [testDeal];
      storage.lastDealTimeByInstanceIndex = testConfig.lastDealTimeByInstanceIndex;
      storage.historyOrders = [testOrder];
      storage.lastHistoryOrderTimeByInstanceIndex = testConfig.lastHistoryOrderTimeByInstanceIndex;
      await fileManager.updateDiskStorage();
      let history = await readHistoryStorageFile();
      _sinon2.default.assert.match(history.deals.items, [testDeal]);
      _sinon2.default.assert.match(history.deals.lastDealTimeByInstanceIndex, testConfig.lastDealTimeByInstanceIndex);
      _sinon2.default.assert.match(history.historyOrders.items, [testOrder]);
      _sinon2.default.assert.match(history.historyOrders.lastHistoryOrderTimeByInstanceIndex, testConfig.lastHistoryOrderTimeByInstanceIndex);
    });
  });

  /**
   * @test {BrowserHistoryManager#deleteStorageFromDisk}
   */
  describe('deleteStorageFromDisk', () => {

    /**
     * @test {BrowserHistoryManager#deleteStorageFromDisk}
     */
    it('should do nothing without error if storage doesnt exist', async () => {
      await deleteDB('metaapi');
      await fileManager.deleteStorageFromDisk();
    });

    /**
     * @test {BrowserHistoryManager#deleteStorageFromDisk}
     */
    it('should do nothing without error if db exists and entries dont exist', async () => {
      await fileManager.deleteStorageFromDisk();
    });

    /**
     * @test {BrowserHistoryManager#deleteStorageFromDisk}
     */
    it('should delete storage', async () => {
      await createTestData([testDeal], [testOrder], testConfig);
      let history = await readHistoryStorageFile();
      _sinon2.default.assert.match(history.deals.items, [testDeal]);
      _sinon2.default.assert.match(history.deals.lastDealTimeByInstanceIndex, testConfig.lastDealTimeByInstanceIndex);
      _sinon2.default.assert.match(history.historyOrders.items, [testOrder]);
      _sinon2.default.assert.match(history.historyOrders.lastHistoryOrderTimeByInstanceIndex, testConfig.lastHistoryOrderTimeByInstanceIndex);
      await fileManager.deleteStorageFromDisk();
      history = await readHistoryStorageFile();
      _sinon2.default.assert.match(history.deals, undefined);
      _sinon2.default.assert.match(history.historyOrders, undefined);
    });
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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