'use strict';

var _should = require('should');

var _should2 = _interopRequireDefault(_should);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _metatraderAccountApi = require('./metatraderAccountApi');

var _metatraderAccountApi2 = _interopRequireDefault(_metatraderAccountApi);

var _metatraderAccount = require('./metatraderAccount');

var _metatraderAccount2 = _interopRequireDefault(_metatraderAccount);

var _errorHandler = require('../clients/errorHandler');

var _index = require('./historyFileManager/index');

var _index2 = _interopRequireDefault(_index);

var _expertAdvisor = require('./expertAdvisor');

var _expertAdvisor2 = _interopRequireDefault(_expertAdvisor);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @test {MetatraderAccountApi}
 * @test {MetatraderAccount}
 */
describe('MetatraderAccountApi', () => {

  let sandbox;
  let api;
  let client = {
    getAccounts: () => {},
    getAccount: () => {},
    getAccountByToken: () => {},
    createAccount: () => {},
    deleteAccount: () => {},
    deployAccount: () => {},
    undeployAccount: () => {},
    redeployAccount: () => {},
    updateAccount: () => {},
    increaseReliability: () => {}
  };
  let eaClient = {
    getExpertAdvisors: () => {},
    getExpertAdvisor: () => {},
    updateExpertAdvisor: () => {},
    uploadExpertAdvisorFile: () => {},
    deleteExpertAdvisor: () => {}
  };
  let metaApiWebsocketClient = {
    addSynchronizationListener: () => {},
    addReconnectListener: () => {},
    subscribe: () => {}
  };
  let connectionRegistry = {
    connect: () => {},
    remove: () => {}
  };

  before(() => {
    api = new _metatraderAccountApi2.default(client, metaApiWebsocketClient, connectionRegistry, eaClient);
    sandbox = _sinon2.default.createSandbox();
  });

  afterEach(() => {
    sandbox.restore();
  });

  /**
   * @test {MetatraderAccountApi#getAccounts}
   */
  it('should retrieve MT accounts', async () => {
    sandbox.stub(client, 'getAccounts').resolves([{ _id: 'id' }]);
    let accounts = await api.getAccounts({ provisioningProfileId: 'profileId' });
    accounts.map(a => a.id).should.match(['id']);
    accounts.forEach(a => (a instanceof _metatraderAccount2.default).should.be.true());
    _sinon2.default.assert.calledWithMatch(client.getAccounts, { provisioningProfileId: 'profileId' });
  });

  /**
   * @test {MetatraderAccountApi#getAccount}
   */
  it('should retrieve MT account by id', async () => {
    sandbox.stub(client, 'getAccount').resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'DISCONNECTED',
      state: 'DEPLOYED',
      type: 'cloud',
      accessToken: '2RUnoH1ldGbnEneCoqRTgI4QO1XOmVzbH5EVoQsA'
    });
    let account = await api.getAccount('id');
    account.id.should.equal('id');
    account.login.should.equal('50194988');
    account.name.should.equal('mt5a');
    account.server.should.equal('ICMarketsSC-Demo');
    account.provisioningProfileId.should.equal('f9ce1f12-e720-4b9a-9477-c2d4cb25f076');
    account.magic.should.equal(123456);
    account.application.should.equal('MetaApi');
    account.connectionStatus.should.equal('DISCONNECTED');
    account.state.should.equal('DEPLOYED');
    account.type.should.equal('cloud');
    account.accessToken.should.equal('2RUnoH1ldGbnEneCoqRTgI4QO1XOmVzbH5EVoQsA');
    (account instanceof _metatraderAccount2.default).should.be.true();
    _sinon2.default.assert.calledWith(client.getAccount, 'id');
  });

  /**
   * @test {MetatraderAccountApi#getAccountByToken}
   */
  it('should retrieve MT account by token', async () => {
    sandbox.stub(client, 'getAccountByToken').resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'DISCONNECTED',
      state: 'DEPLOYED',
      type: 'cloud',
      accessToken: '2RUnoH1ldGbnEneCoqRTgI4QO1XOmVzbH5EVoQsA'
    });
    let account = await api.getAccountByToken();
    account.id.should.equal('id');
    account.login.should.equal('50194988');
    account.name.should.equal('mt5a');
    account.server.should.equal('ICMarketsSC-Demo');
    account.provisioningProfileId.should.equal('f9ce1f12-e720-4b9a-9477-c2d4cb25f076');
    account.magic.should.equal(123456);
    account.application.should.equal('MetaApi');
    account.connectionStatus.should.equal('DISCONNECTED');
    account.state.should.equal('DEPLOYED');
    account.type.should.equal('cloud');
    account.accessToken.should.equal('2RUnoH1ldGbnEneCoqRTgI4QO1XOmVzbH5EVoQsA');
    (account instanceof _metatraderAccount2.default).should.be.true();
    _sinon2.default.assert.calledWith(client.getAccountByToken);
  });

  /**
   * @test {MetatraderAccountApi#createAccount}
   */
  it('should create MT account', async () => {
    sandbox.stub(client, 'createAccount').resolves({ id: 'id' });
    sandbox.stub(client, 'getAccount').resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'DISCONNECTED',
      state: 'DEPLOYED',
      type: 'cloud',
      accessToken: '2RUnoH1ldGbnEneCoqRTgI4QO1XOmVzbH5EVoQsA'
    });
    let newAccountData = {
      login: '50194988',
      password: 'Test1234',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      type: 'cloud',
      accessToken: 'NyV5no9TMffJyUts2FjI80wly0so3rVCz4xOqiDx'
    };
    let account = await api.createAccount(newAccountData);
    account.id.should.equal('id');
    account.login.should.equal('50194988');
    account.name.should.equal('mt5a');
    account.server.should.equal('ICMarketsSC-Demo');
    account.provisioningProfileId.should.equal('f9ce1f12-e720-4b9a-9477-c2d4cb25f076');
    account.magic.should.equal(123456);
    account.application.should.equal('MetaApi');
    account.connectionStatus.should.equal('DISCONNECTED');
    account.state.should.equal('DEPLOYED');
    account.type.should.equal('cloud');
    account.accessToken.should.equal('2RUnoH1ldGbnEneCoqRTgI4QO1XOmVzbH5EVoQsA');
    (account instanceof _metatraderAccount2.default).should.be.true();
    _sinon2.default.assert.calledWith(client.createAccount, newAccountData);
    _sinon2.default.assert.calledWith(client.getAccount, 'id');
  });

  /**
   * @test {MetatraderAccount#reload}
   */
  it('should reload MT account', async () => {
    sandbox.stub(client, 'getAccount').onFirstCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'DISCONNECTED',
      state: 'DEPLOYING',
      type: 'cloud'
    }).onSecondCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'CONNECTED',
      state: 'DEPLOYED',
      type: 'cloud'
    });
    let account = await api.getAccount('id');
    await account.reload();
    account.connectionStatus.should.equal('CONNECTED');
    account.state.should.equal('DEPLOYED');
    _sinon2.default.assert.calledWith(client.getAccount, 'id');
    _sinon2.default.assert.calledTwice(client.getAccount);
  });

  /**
   * @test {MetatraderAccount#remove}
   */
  it('should remove MT account', async () => {
    sandbox.stub(connectionRegistry, 'remove').resolves();
    sandbox.stub(client, 'getAccount').onFirstCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'CONNECTED',
      state: 'DEPLOYED',
      type: 'cloud'
    }).onSecondCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'CONNECTED',
      state: 'DELETING',
      type: 'cloud'
    });
    sandbox.stub(client, 'deleteAccount').resolves();
    sandbox.stub(_index2.default.prototype, 'deleteStorageFromDisk').returns();
    let account = await api.getAccount('id');
    await account.remove();
    _sinon2.default.assert.calledWith(connectionRegistry.remove, 'id');
    _sinon2.default.assert.calledOnce(_index2.default.prototype.deleteStorageFromDisk);
    account.state.should.equal('DELETING');
    _sinon2.default.assert.calledWith(client.deleteAccount, 'id');
    _sinon2.default.assert.calledWith(client.getAccount, 'id');
    _sinon2.default.assert.calledTwice(client.getAccount);
  });

  /**
   * @test {MetatraderAccount#deploy}
   */
  it('should deploy MT account', async () => {
    sandbox.stub(client, 'getAccount').onFirstCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'DISCONNECTED',
      state: 'UNDEPLOYED',
      type: 'cloud'
    }).onSecondCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'CONNECTED',
      state: 'DEPLOYING',
      type: 'cloud'
    });
    sandbox.stub(client, 'deployAccount').resolves();
    let account = await api.getAccount('id');
    await account.deploy();
    account.state.should.equal('DEPLOYING');
    _sinon2.default.assert.calledWith(client.deployAccount, 'id');
    _sinon2.default.assert.calledWith(client.getAccount, 'id');
    _sinon2.default.assert.calledTwice(client.getAccount);
  });

  /**
   * @test {MetatraderAccount#undeploy}
   */
  it('should undeploy MT account', async () => {
    sandbox.stub(connectionRegistry, 'remove').resolves();
    sandbox.stub(client, 'getAccount').onFirstCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'DISCONNECTED',
      state: 'DEPLOYED',
      type: 'cloud'
    }).onSecondCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'CONNECTED',
      state: 'UNDEPLOYING',
      type: 'cloud'
    });
    sandbox.stub(client, 'undeployAccount').resolves();
    let account = await api.getAccount('id');
    await account.undeploy();
    _sinon2.default.assert.calledWith(connectionRegistry.remove, 'id');
    account.state.should.equal('UNDEPLOYING');
    _sinon2.default.assert.calledWith(client.undeployAccount, 'id');
    _sinon2.default.assert.calledWith(client.getAccount, 'id');
    _sinon2.default.assert.calledTwice(client.getAccount);
  });

  /**
   * @test {MetatraderAccount#redeploy}
   */
  it('should redeploy MT account', async () => {
    sandbox.stub(client, 'getAccount').onFirstCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'DISCONNECTED',
      state: 'DEPLOYED',
      type: 'cloud'
    }).onSecondCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'CONNECTED',
      state: 'UNDEPLOYING',
      type: 'cloud'
    });
    sandbox.stub(client, 'redeployAccount').resolves();
    let account = await api.getAccount('id');
    await account.redeploy();
    account.state.should.equal('UNDEPLOYING');
    _sinon2.default.assert.calledWith(client.redeployAccount, 'id');
    _sinon2.default.assert.calledWith(client.getAccount, 'id');
    _sinon2.default.assert.calledTwice(client.getAccount);
  });

  /**
   * @test {MetatraderAccount#increaseReliability}
   */
  it('should increase MT account reliability', async () => {
    sandbox.stub(client, 'getAccount').onFirstCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'DISCONNECTED',
      state: 'DEPLOYED',
      type: 'cloud'
    }).onSecondCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'CONNECTED',
      state: 'UNDEPLOYING',
      type: 'cloud',
      reliability: 'high'
    });
    sandbox.stub(client, 'increaseReliability').resolves();
    let account = await api.getAccount('id');
    await account.increaseReliability();
    account.reliability.should.equal('high');
    _sinon2.default.assert.calledWith(client.increaseReliability, 'id');
    _sinon2.default.assert.calledWith(client.getAccount, 'id');
    _sinon2.default.assert.calledTwice(client.getAccount);
  });

  describe('MetatraderAccount.waitDeployed', () => {

    /**
     * @test {MetatraderAccount#waitDeployed}
     */
    it('should wait for deployment', async () => {
      let deployingAccount = {
        _id: 'id',
        login: '50194988',
        name: 'mt5a',
        server: 'ICMarketsSC-Demo',
        provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
        magic: 123456,
        application: 'MetaApi',
        connectionStatus: 'DISCONNECTED',
        state: 'DEPLOYING',
        type: 'cloud'
      };
      sandbox.stub(client, 'getAccount').onFirstCall().resolves(deployingAccount).onSecondCall().resolves(deployingAccount).onThirdCall().resolves({
        _id: 'id',
        login: '50194988',
        name: 'mt5a',
        server: 'ICMarketsSC-Demo',
        provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
        magic: 123456,
        application: 'MetaApi',
        connectionStatus: 'CONNECTED',
        state: 'DEPLOYED',
        type: 'cloud'
      });
      let account = await api.getAccount('id');
      await account.waitDeployed(1, 50);
      account.state.should.equal('DEPLOYED');
      _sinon2.default.assert.calledWith(client.getAccount, 'id');
      _sinon2.default.assert.calledThrice(client.getAccount);
    });

    /**
     * @test {MetatraderAccount#waitDeployed}
     */
    it('should time out waiting for deployment', async () => {
      let deployingAccount = {
        _id: 'id',
        login: '50194988',
        name: 'mt5a',
        server: 'ICMarketsSC-Demo',
        provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
        magic: 123456,
        application: 'MetaApi',
        connectionStatus: 'DISCONNECTED',
        state: 'DEPLOYING',
        type: 'cloud'
      };
      sandbox.stub(client, 'getAccount').resolves(deployingAccount);
      let account = await api.getAccount('id');
      try {
        await account.waitDeployed(1, 50);
        throw new Error('TimeoutError is expected');
      } catch (err) {
        err.name.should.equal('TimeoutError');
        account.state.should.equal('DEPLOYING');
      }
      _sinon2.default.assert.calledWith(client.getAccount, 'id');
    });
  });

  describe('MetatraderAccount.waitUndeployed', () => {

    /**
     * @test {MetatraderAccount#waitUndeployed}
     */
    it('should wait for undeployment', async () => {
      let undeployingAccount = {
        _id: 'id',
        login: '50194988',
        name: 'mt5a',
        server: 'ICMarketsSC-Demo',
        provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
        magic: 123456,
        application: 'MetaApi',
        connectionStatus: 'DISCONNECTED',
        state: 'UNDEPLOYING',
        type: 'cloud'
      };
      sandbox.stub(client, 'getAccount').onFirstCall().resolves(undeployingAccount).onSecondCall().resolves(undeployingAccount).onThirdCall().resolves({
        _id: 'id',
        login: '50194988',
        name: 'mt5a',
        server: 'ICMarketsSC-Demo',
        provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
        magic: 123456,
        application: 'MetaApi',
        connectionStatus: 'CONNECTED',
        state: 'UNDEPLOYED',
        type: 'cloud'
      });
      let account = await api.getAccount('id');
      await account.waitUndeployed(1, 50);
      account.state.should.equal('UNDEPLOYED');
      _sinon2.default.assert.calledWith(client.getAccount, 'id');
      _sinon2.default.assert.calledThrice(client.getAccount);
    });

    /**
     * @test {MetatraderAccount#waitUndeployed}
     */
    it('should time out waiting for undeployment', async () => {
      let undeployingAccount = {
        _id: 'id',
        login: '50194988',
        name: 'mt5a',
        server: 'ICMarketsSC-Demo',
        provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
        magic: 123456,
        application: 'MetaApi',
        connectionStatus: 'DISCONNECTED',
        state: 'UNDEPLOYING',
        type: 'cloud'
      };
      sandbox.stub(client, 'getAccount').resolves(undeployingAccount);
      let account = await api.getAccount('id');
      try {
        await account.waitUndeployed(1, 50);
        throw new Error('TimeoutError is expected');
      } catch (err) {
        err.name.should.equal('TimeoutError');
        account.state.should.equal('UNDEPLOYING');
      }
      _sinon2.default.assert.calledWith(client.getAccount, 'id');
    });
  });

  describe('MetatraderAccount.waitRemoved', () => {

    /**
     * @test {MetatraderAccount#waitRemoved}
     */
    it('should wait until removed', async () => {
      let deletingAccount = {
        _id: 'id',
        login: '50194988',
        name: 'mt5a',
        server: 'ICMarketsSC-Demo',
        provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
        magic: 123456,
        application: 'MetaApi',
        connectionStatus: 'DISCONNECTED',
        state: 'DELETING',
        type: 'cloud'
      };
      sandbox.stub(client, 'getAccount').onFirstCall().resolves(deletingAccount).onSecondCall().resolves(deletingAccount).onThirdCall().throws(new _errorHandler.NotFoundError());
      let account = await api.getAccount('id');
      await account.waitRemoved(1, 50);
      _sinon2.default.assert.calledWith(client.getAccount, 'id');
      _sinon2.default.assert.calledThrice(client.getAccount);
    });

    /**
     * @test {MetatraderAccount#waitRemoved}
     */
    it('should time out waiting until removed', async () => {
      let deletingAccount = {
        _id: 'id',
        login: '50194988',
        name: 'mt5a',
        server: 'ICMarketsSC-Demo',
        provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
        magic: 123456,
        application: 'MetaApi',
        connectionStatus: 'DISCONNECTED',
        state: 'DELETING',
        type: 'cloud'
      };
      sandbox.stub(client, 'getAccount').resolves(deletingAccount);
      let account = await api.getAccount('id');
      try {
        await account.waitRemoved(1, 50);
        throw new Error('TimeoutError is expected');
      } catch (err) {
        err.name.should.equal('TimeoutError');
      }
      _sinon2.default.assert.calledWith(client.getAccount, 'id');
    });
  });

  describe('MetatraderAccount.waitConnected', () => {

    /**
     * @test {MetatraderAccount#waitConnected}
     */
    it('should wait until broker connection', async () => {
      let disconnectedAccount = {
        _id: 'id',
        login: '50194988',
        name: 'mt5a',
        server: 'ICMarketsSC-Demo',
        provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
        magic: 123456,
        application: 'MetaApi',
        connectionStatus: 'DISCONNECTED',
        state: 'DEPLOYED',
        type: 'cloud'
      };
      sandbox.stub(client, 'getAccount').onFirstCall().resolves(disconnectedAccount).onSecondCall().resolves(disconnectedAccount).onThirdCall().resolves({
        _id: 'id',
        login: '50194988',
        name: 'mt5a',
        server: 'ICMarketsSC-Demo',
        provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
        magic: 123456,
        application: 'MetaApi',
        connectionStatus: 'CONNECTED',
        state: 'DEPLOYED',
        type: 'cloud'
      });
      let account = await api.getAccount('id');
      await account.waitConnected(1, 50);
      account.connectionStatus.should.equal('CONNECTED');
      _sinon2.default.assert.calledWith(client.getAccount, 'id');
      _sinon2.default.assert.calledThrice(client.getAccount);
    });

    /**
     * @test {MetatraderAccount#waitConnected}
     */
    it('should time out waiting for broker connection', async () => {
      let disconnectedAccount = {
        _id: 'id',
        login: '50194988',
        name: 'mt5a',
        server: 'ICMarketsSC-Demo',
        provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
        magic: 123456,
        application: 'MetaApi',
        connectionStatus: 'DISCONNECTED',
        state: 'DEPLOYED',
        type: 'cloud'
      };
      sandbox.stub(client, 'getAccount').resolves(disconnectedAccount);
      let account = await api.getAccount('id');
      try {
        await account.waitConnected(1, 50);
        throw new Error('TimeoutError is expected');
      } catch (err) {
        err.name.should.equal('TimeoutError');
        account.connectionStatus.should.equal('DISCONNECTED');
      }
      _sinon2.default.assert.calledWith(client.getAccount, 'id');
    });
  });

  /**
   * @test {MetatraderAccount#connect}
   */
  it('should connect to an MT terminal', async () => {
    sandbox.stub(metaApiWebsocketClient, 'addSynchronizationListener').returns();
    sandbox.stub(metaApiWebsocketClient, 'subscribe').resolves();
    sandbox.stub(client, 'getAccount').resolves({ _id: 'id' });
    let account = await api.getAccount();
    let storage = {
      lastHistoryOrderTime: () => new Date('2020-01-01T00:00:00.000Z'),
      lastDealTime: () => new Date('2020-01-02T00:00:00.000Z'),
      loadDataFromDisk: () => ({ deals: [], historyOrders: [] })
    };
    sandbox.spy(connectionRegistry, 'connect');
    let connection = account.getStreamingConnection(storage);
    _sinon2.default.assert.calledWith(connectionRegistry.connect, account, storage);
  });

  /**
   * @test {MetatraderAccount#update}
   */
  it('should update MT account', async () => {
    sandbox.stub(client, 'getAccount').onFirstCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a',
      server: 'ICMarketsSC-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'CONNECTED',
      state: 'DEPLOYED',
      type: 'cloud'
    }).onSecondCall().resolves({
      _id: 'id',
      login: '50194988',
      name: 'mt5a__',
      server: 'OtherMarkets-Demo',
      provisioningProfileId: 'f9ce1f12-e720-4b9a-9477-c2d4cb25f076',
      magic: 123456,
      application: 'MetaApi',
      connectionStatus: 'CONNECTED',
      state: 'DEPLOYED',
      type: 'cloud'
    });
    sandbox.stub(client, 'updateAccount').resolves();
    let account = await api.getAccount('id');
    await account.update({
      name: 'mt5a__',
      password: 'moreSecurePass',
      server: 'OtherMarkets-Demo'
    });
    account.name.should.equal('mt5a__');
    account.server.should.equal('OtherMarkets-Demo');
    _sinon2.default.assert.calledWith(client.updateAccount, 'id', {
      name: 'mt5a__',
      password: 'moreSecurePass',
      server: 'OtherMarkets-Demo'
    });
    _sinon2.default.assert.calledWith(client.getAccount, 'id');
    _sinon2.default.assert.calledTwice(client.getAccount);
  });

  /**
   * @test {MetatraderAccount#getExpertAdvisors}
   */
  it('should retrieve expert advisors', async () => {
    sandbox.stub(client, 'getAccount').resolves({ _id: 'id', version: 4, type: 'cloud-g1' });
    sandbox.stub(eaClient, 'getExpertAdvisors').resolves([{ expertId: 'ea' }]);
    const account = await api.getAccount();
    const experts = await account.getExpertAdvisors();
    experts.map(e => e.expertId).should.match(['ea']);
    experts.forEach(e => (e instanceof _expertAdvisor2.default).should.be.true());
    _sinon2.default.assert.calledWithMatch(eaClient.getExpertAdvisors, 'id');
  });

  /**
   * @test {MetatraderAccount#getExpertAdvisor}
   */
  it('should retrieve expert advisor by expert id', async () => {
    sandbox.stub(client, 'getAccount').resolves({ _id: 'id', version: 4, type: 'cloud-g1' });
    sandbox.stub(eaClient, 'getExpertAdvisor').resolves({
      expertId: 'ea',
      period: '1H',
      symbol: 'EURUSD',
      fileUploaded: false
    });
    const account = await api.getAccount('id');
    const expert = await account.getExpertAdvisor('ea');
    expert.expertId.should.match('ea');
    expert.period.should.match('1H');
    expert.symbol.should.match('EURUSD');
    expert.fileUploaded.should.be.false();
    (expert instanceof _expertAdvisor2.default).should.be.true();
    _sinon2.default.assert.calledWithMatch(eaClient.getExpertAdvisor, 'id', 'ea');
  });

  /**
   * @test {MetatraderAccount#getExpertAdvisor}
   */
  it('should validate account version', async () => {
    sandbox.stub(client, 'getAccount').resolves({ _id: 'id', version: 5, type: 'cloud-g1' });
    sandbox.stub(eaClient, 'getExpertAdvisors').resolves([{
      expertId: 'ea',
      period: '1H',
      symbol: 'EURUSD',
      fileUploaded: false
    }]);
    sandbox.stub(eaClient, 'getExpertAdvisor').resolves({
      expertId: 'ea',
      period: '1H',
      symbol: 'EURUSD',
      fileUploaded: false
    });
    sandbox.stub(eaClient, 'updateExpertAdvisor').resolves();
    let newExpertAdvisor = {
      period: '1H',
      symbol: 'EURUSD',
      preset: 'preset'
    };
    const account = await api.getAccount('id');
    await (0, _should2.default)(account.getExpertAdvisors()).rejected();
    await (0, _should2.default)(account.getExpertAdvisor('ea')).rejected();
    await (0, _should2.default)(account.createExpertAdvisor('ea', newExpertAdvisor)).rejected();
  });

  /**
   * @test {MetatraderAccount#getExpertAdvisor}
   */
  it('should validate account type', async () => {
    sandbox.stub(client, 'getAccount').resolves({ _id: 'id', version: 4, type: 'cloud-g2' });
    sandbox.stub(eaClient, 'getExpertAdvisors').resolves([{
      expertId: 'ea',
      period: '1H',
      symbol: 'EURUSD',
      fileUploaded: false
    }]);
    sandbox.stub(eaClient, 'getExpertAdvisor').resolves({
      expertId: 'ea',
      period: '1H',
      symbol: 'EURUSD',
      fileUploaded: false
    });
    sandbox.stub(eaClient, 'updateExpertAdvisor').resolves();
    let newExpertAdvisor = {
      period: '1H',
      symbol: 'EURUSD',
      preset: 'preset'
    };
    const account = await api.getAccount('id');
    await (0, _should2.default)(account.getExpertAdvisors()).rejected();
    await (0, _should2.default)(account.getExpertAdvisor('ea')).rejected();
    await (0, _should2.default)(account.createExpertAdvisor('ea', newExpertAdvisor)).rejected();
  });

  /**
   * @test {MetatraderAccount#createExpertAdvisor}
   */
  it('should create expert advisor', async () => {
    sandbox.stub(client, 'getAccount').resolves({ _id: 'id', version: 4, type: 'cloud-g1' });
    sandbox.stub(eaClient, 'updateExpertAdvisor').resolves();
    sandbox.stub(eaClient, 'getExpertAdvisor').resolves({
      expertId: 'ea',
      period: '1H',
      symbol: 'EURUSD',
      fileUploaded: false
    });
    let newExpertAdvisor = {
      period: '1H',
      symbol: 'EURUSD',
      preset: 'preset'
    };
    const account = await api.getAccount('id');
    const expert = await account.createExpertAdvisor('ea', newExpertAdvisor);
    expert.expertId.should.match('ea');
    expert.period.should.match('1H');
    expert.symbol.should.match('EURUSD');
    expert.fileUploaded.should.be.false();
    (expert instanceof _expertAdvisor2.default).should.be.true();
    _sinon2.default.assert.calledWith(eaClient.updateExpertAdvisor, 'id', 'ea', newExpertAdvisor);
    _sinon2.default.assert.calledWith(eaClient.getExpertAdvisor, 'id', 'ea');
  });

  /**
   * @test {ExpertAdvisor#reload}
   */
  it('should reload expert advisor', async () => {
    sandbox.stub(client, 'getAccount').resolves({ _id: 'id', version: 4, type: 'cloud-g1' });
    sandbox.stub(eaClient, 'getExpertAdvisor').onFirstCall().resolves({
      expertId: 'ea',
      period: '1H',
      symbol: 'EURUSD',
      fileUploaded: false
    }).onSecondCall().resolves({
      expertId: 'ea',
      period: '4H',
      symbol: 'EURUSD',
      fileUploaded: false
    });
    const account = await api.getAccount('id');
    const expert = await account.getExpertAdvisor('ea');
    await expert.reload();
    expert.period.should.eql('4H');
    _sinon2.default.assert.calledWith(eaClient.getExpertAdvisor, 'id', 'ea');
    _sinon2.default.assert.calledTwice(eaClient.getExpertAdvisor);
  });

  /**
   * @test {ExpertAdvisor#update}
   */
  it('should update expert advisor', async () => {
    sandbox.stub(client, 'getAccount').resolves({ _id: 'id', version: 4, type: 'cloud-g1' });
    sandbox.stub(eaClient, 'getExpertAdvisor').onFirstCall().resolves({
      expertId: 'ea',
      period: '1H',
      symbol: 'EURUSD',
      fileUploaded: false
    }).onSecondCall().resolves({
      expertId: 'ea',
      period: '4H',
      symbol: 'EURUSD',
      fileUploaded: false
    });
    let newExpertAdvisor = {
      period: '4H',
      symbol: 'EURUSD',
      preset: 'preset'
    };
    sandbox.stub(eaClient, 'updateExpertAdvisor').resolves();
    const account = await api.getAccount('id');
    const expert = await account.getExpertAdvisor('ea');
    await expert.update(newExpertAdvisor);
    expert.period.should.eql('4H');
    _sinon2.default.assert.calledWith(eaClient.updateExpertAdvisor, 'id', 'ea', newExpertAdvisor);
    _sinon2.default.assert.calledTwice(eaClient.getExpertAdvisor);
    _sinon2.default.assert.calledWith(eaClient.getExpertAdvisor, 'id', 'ea');
  });

  /**
   * @test {ExpertAdvisor#uploadFile}
   */
  it('should upload expert advisor file', async () => {
    sandbox.stub(client, 'getAccount').resolves({ _id: 'id', version: 4, type: 'cloud-g1' });
    sandbox.stub(eaClient, 'getExpertAdvisor').onFirstCall().resolves({
      expertId: 'ea',
      period: '1H',
      symbol: 'EURUSD',
      fileUploaded: false
    }).onSecondCall().resolves({
      expertId: 'ea',
      period: '4H',
      symbol: 'EURUSD',
      fileUploaded: true
    });
    sandbox.stub(eaClient, 'uploadExpertAdvisorFile').resolves();
    const account = await api.getAccount('id');
    const expert = await account.getExpertAdvisor('ea');
    await expert.uploadFile('/path/to/file');
    expert.fileUploaded.should.be.true();
    _sinon2.default.assert.calledWith(eaClient.uploadExpertAdvisorFile, 'id', 'ea', '/path/to/file');
    _sinon2.default.assert.calledTwice(eaClient.getExpertAdvisor);
    _sinon2.default.assert.calledWith(eaClient.getExpertAdvisor, 'id', 'ea');
  });

  /**
   * @test {ExpertAdvisor#remove}
   */
  it('should remove expert advisor', async () => {
    sandbox.stub(client, 'getAccount').resolves({ _id: 'id', version: 4, type: 'cloud-g1' });
    sandbox.stub(eaClient, 'getExpertAdvisor').resolves({
      expertId: 'ea',
      period: '1H',
      symbol: 'EURUSD',
      fileUploaded: false
    });
    sandbox.stub(eaClient, 'deleteExpertAdvisor').resolves({ _id: 'id' });
    const account = await api.getAccount('id');
    const expert = await account.getExpertAdvisor('ea');
    await expert.remove();
    _sinon2.default.assert.calledWith(eaClient.deleteExpertAdvisor, 'id', 'ea');
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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