'use strict';

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _should = require('should');

var _should2 = _interopRequireDefault(_should);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _streamingMetaApiConnection = require('./streamingMetaApiConnection');

var _streamingMetaApiConnection2 = _interopRequireDefault(_streamingMetaApiConnection);

var _notSynchronizedError = require('../clients/metaApi/notSynchronizedError');

var _notSynchronizedError2 = _interopRequireDefault(_notSynchronizedError);

var _randomstring = require('randomstring');

var _randomstring2 = _interopRequireDefault(_randomstring);

var _index = require('./historyFileManager/index');

var _index2 = _interopRequireDefault(_index);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @test {MetaApiConnection}
 */
// eslint-disable-next-line max-statements
describe('StreamingMetaApiConnection', () => {

  let sandbox;
  let api;
  let account;
  let clock;
  let client = {
    getAccountInformation: () => {},
    getPositions: () => {},
    getPosition: () => {},
    getOrders: () => {},
    getOrder: () => {},
    getHistoryOrdersByTicket: () => {},
    getHistoryOrdersByPosition: () => {},
    getHistoryOrdersByTimeRange: () => {},
    getDealsByTicket: () => {},
    getDealsByPosition: () => {},
    getDealsByTimeRange: () => {},
    removeHistory: () => {},
    removeApplication: () => {},
    trade: () => {},
    reconnect: () => {},
    synchronize: () => true,
    ensureSubscribe: () => {},
    subscribeToMarketData: () => {},
    unsubscribeFromMarketData: () => {},
    addSynchronizationListener: () => {},
    addReconnectListener: () => {},
    removeSynchronizationListener: () => {},
    removeReconnectListener: () => {},
    getSymbols: () => {},
    getSymbolSpecification: () => {},
    getSymbolPrice: () => {},
    getCandle: () => {},
    getTick: () => {},
    getBook: () => {},
    saveUptime: () => {},
    waitSynchronized: () => {},
    unsubscribe: () => {},
    refreshMarketDataSubscriptions: () => {}
  };

  let connectionRegistry = {
    connect: () => {},
    remove: () => {},
    application: 'MetaApi'
  };

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(() => {
    account = {
      id: 'accountId',
      state: 'DEPLOYED',
      reload: () => {}
    };
    sandbox.stub(_index2.default.prototype, 'startUpdateJob').returns();
    api = new _streamingMetaApiConnection2.default(client, account, undefined, connectionRegistry, 0, {
      minDelayInSeconds: 1,
      maxDelayInSeconds: 1
    });
    sandbox.stub(api._terminalState, 'specification').withArgs('EURUSD').returns({ symbol: 'EURUSD' });
    clock = _sinon2.default.useFakeTimers({
      shouldAdvanceTime: true
    });
  });

  afterEach(() => {
    clock.restore();
    sandbox.restore();
  });

  /**
   * @test {MetaApiConnection#removeHistory}
   */
  it('should remove history', async () => {
    sandbox.stub(client, 'removeHistory').resolves();
    sandbox.stub(api.historyStorage, 'clear').resolves();
    await api.removeHistory('app');
    _sinon2.default.assert.calledWith(client.removeHistory, 'accountId', 'app');
    _sinon2.default.assert.calledOnce(api.historyStorage.clear);
  });

  /**
   * @test {MetaApiConnection#removeApplication}
   */
  it('should remove application', async () => {
    sandbox.stub(client, 'removeApplication').resolves();
    sandbox.stub(api.historyStorage, 'clear').resolves();
    await api.removeApplication();
    _sinon2.default.assert.calledWith(client.removeApplication, 'accountId');
    _sinon2.default.assert.calledOnce(api.historyStorage.clear);
  });

  /**
   * @test {MetaApiConnection#createMarketBuyOrder}
   */
  it('should create market buy order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createMarketBuyOrder('GBPUSD', 0.07, 0.9, 2.0, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_BUY', symbol: 'GBPUSD',
      volume: 0.07, stopLoss: 0.9, takeProfit: 2.0, comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }));
  });

  /**
   * @test {MetaApiConnection#createMarketBuyOrder}
   */
  it('should create market buy order with relative SL/TP', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createMarketBuyOrder('GBPUSD', 0.07, { value: 0.1, units: 'RELATIVE_PRICE' }, { value: 2000, units: 'RELATIVE_POINTS' }, { comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_BUY', symbol: 'GBPUSD',
      volume: 0.07, stopLoss: 0.1, stopLossUnits: 'RELATIVE_PRICE', takeProfit: 2000,
      takeProfitUnits: 'RELATIVE_POINTS', comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }));
  });

  /**
   * @test {MetaApiConnection#createMarketSellOrder}
   */
  it('should create market sell order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createMarketSellOrder('GBPUSD', 0.07, 2.0, 0.9, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_SELL', symbol: 'GBPUSD',
      volume: 0.07, stopLoss: 2.0, takeProfit: 0.9, comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }));
  });

  /**
   * @test {MetaApiConnection#createLimitBuyOrder}
   */
  it('should create limit buy order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createLimitBuyOrder('GBPUSD', 0.07, 1.0, 0.9, 2.0, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_BUY_LIMIT',
      symbol: 'GBPUSD', volume: 0.07, openPrice: 1.0, stopLoss: 0.9, takeProfit: 2.0, comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' }));
  });

  /**
   * @test {MetaApiConnection#createLimitSellOrder}
   */
  it('should create limit sell order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createLimitSellOrder('GBPUSD', 0.07, 1.5, 2.0, 0.9, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_SELL_LIMIT',
      symbol: 'GBPUSD', volume: 0.07, openPrice: 1.5, stopLoss: 2.0, takeProfit: 0.9, comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' }));
  });

  /**
   * @test {MetaApiConnection#createStopBuyOrder}
   */
  it('should create stop buy order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createStopBuyOrder('GBPUSD', 0.07, 1.5, 0.9, 2.0, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_BUY_STOP',
      symbol: 'GBPUSD', volume: 0.07, openPrice: 1.5, stopLoss: 0.9, takeProfit: 2.0, comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' }));
  });

  /**
   * @test {MetaApiConnection#createStopSellOrder}
   */
  it('should create stop sell order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createStopSellOrder('GBPUSD', 0.07, 1.0, 2.0, 0.9, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_SELL_STOP',
      symbol: 'GBPUSD', volume: 0.07, openPrice: 1.0, stopLoss: 2.0, takeProfit: 0.9, comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' }));
  });

  /**
   * @test {MetaApiConnection#createStopLimitBuyOrder}
   */
  it('should create stop limit buy order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: 46870472
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createStopLimitBuyOrder('GBPUSD', 0.07, 1.5, 1.4, 0.9, 2.0, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_BUY_STOP_LIMIT',
      symbol: 'GBPUSD', volume: 0.07, openPrice: 1.5, stopLimitPrice: 1.4, stopLoss: 0.9, takeProfit: 2.0,
      comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }));
  });

  /**
   * @test {MetaApiConnection#createStopLimitSellOrder}
   */
  it('should create stop limit sell order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.createStopLimitSellOrder('GBPUSD', 0.07, 1.0, 1.1, 2.0, 0.9, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_TYPE_SELL_STOP_LIMIT',
      symbol: 'GBPUSD', volume: 0.07, openPrice: 1.0, stopLimitPrice: 1.1, stopLoss: 2.0, takeProfit: 0.9,
      comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }));
  });

  /**
   * @test {MetaApiConnection#modifyPosition}
   */
  it('should modify position', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      positionId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.modifyPosition('46870472', 2.0, 0.9);
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'POSITION_MODIFY',
      positionId: '46870472', stopLoss: 2.0, takeProfit: 0.9 }));
  });

  /**
   * @test {MetaApiConnection#closePositionPartially}
   */
  it('should close position partially', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      positionId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.closePositionPartially('46870472', 0.9, { comment: 'comment',
      clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'POSITION_PARTIAL',
      positionId: '46870472', volume: 0.9, comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }));
  });

  /**
   * @test {MetaApiConnection#closePosition}
   */
  it('should close position', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      positionId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.closePosition('46870472', { comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'POSITION_CLOSE_ID',
      positionId: '46870472', comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }));
  });

  /**
   * @test {MetaApiConnection#closeBy}
   */
  it('should close position by an opposite one', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      positionId: '46870472',
      closeByPositionId: '46870482'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.closeBy('46870472', '46870482', { comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'POSITION_CLOSE_BY',
      positionId: '46870472', closeByPositionId: '46870482', comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }));
  });

  /**
   * @test {MetaApiConnection#closePositionsBySymbol}
   */
  it('should close positions by symbol', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      positionId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.closePositionsBySymbol('EURUSD', { comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' });
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'POSITIONS_CLOSE_SYMBOL',
      symbol: 'EURUSD', comment: 'comment', clientId: 'TE_GBPUSD_7hyINWqAlE' }));
  });

  /**
   * @test {MetaApiConnection#modifyOrder}
   */
  it('should modify order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.modifyOrder('46870472', 1.0, 2.0, 0.9);
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_MODIFY', orderId: '46870472',
      openPrice: 1.0, stopLoss: 2.0, takeProfit: 0.9 }));
  });

  /**
   * @test {MetaApiConnection#cancelOrder}
   */
  it('should cancel order', async () => {
    let tradeResult = {
      error: 10009,
      description: 'TRADE_RETCODE_DONE',
      orderId: '46870472'
    };
    sandbox.stub(client, 'trade').resolves(tradeResult);
    let actual = await api.cancelOrder('46870472');
    actual.should.match(tradeResult);
    _sinon2.default.assert.calledWith(client.trade, 'accountId', _sinon2.default.match({ actionType: 'ORDER_CANCEL', orderId: '46870472' }));
  });

  /**
   * @test {MetaApiConnection#subscribe}
   */
  describe('ensure subscribe', () => {

    /**
     * @test {MetaApiConnection#subscribe}
     */
    it('should subscribe to terminal', async () => {
      sandbox.stub(client, 'ensureSubscribe').resolves();
      await api.subscribe();
      _sinon2.default.assert.calledWith(client.ensureSubscribe, 'accountId');
    });
  });

  /**
   * @test {MetaApiConnection#synchronize}
   */
  it('should not subscribe if connection is closed', async () => {
    const ensureSubscribeStub = sandbox.stub(client, 'ensureSubscribe').resolves();
    await api.close();
    await api.subscribe();
    _sinon2.default.assert.notCalled(ensureSubscribeStub);
  });

  /**
   * @test {MetaApiConnection#synchronize}
   */
  it('should synchronize state with terminal', async () => {
    sandbox.stub(client, 'synchronize').resolves();
    sandbox.stub(_randomstring2.default, 'generate').returns('synchronizationId');
    api = new _streamingMetaApiConnection2.default(client, { id: 'accountId' }, undefined, connectionRegistry);
    api.historyStorage.onHistoryOrderAdded('1:ps-mpa-1', { doneTime: new Date('2020-01-01T00:00:00.000Z') });
    api.historyStorage.onDealAdded('1:ps-mpa-1', { time: new Date('2020-01-02T00:00:00.000Z') });
    await api.synchronize('1:ps-mpa-1');
    _sinon2.default.assert.calledWith(client.synchronize, 'accountId', 1, 'ps-mpa-1', 'synchronizationId', new Date('2020-01-01T00:00:00.000Z'), new Date('2020-01-02T00:00:00.000Z'), null, null, null);
  });

  /**
   * @test {MetaApiConnection#synchronize}
   */
  it('should synchronize state with terminal from specified time', async () => {
    sandbox.stub(client, 'synchronize').resolves();
    sandbox.stub(_randomstring2.default, 'generate').returns('synchronizationId');
    api = new _streamingMetaApiConnection2.default(client, { id: 'accountId' }, undefined, connectionRegistry, new Date('2020-10-07T00:00:00.000Z'));
    api.historyStorage.onHistoryOrderAdded('1:ps-mpa-1', { doneTime: new Date('2020-01-01T00:00:00.000Z') });
    api.historyStorage.onDealAdded('1:ps-mpa-1', { time: new Date('2020-01-02T00:00:00.000Z') });
    await api.synchronize('1:ps-mpa-1');
    _sinon2.default.assert.calledWith(client.synchronize, 'accountId', 1, 'ps-mpa-1', 'synchronizationId', new Date('2020-10-07T00:00:00.000Z'), new Date('2020-10-07T00:00:00.000Z'), null, null, null);
  });

  /**
   * @test {MetaApiConnection#subscribeToMarketData}
   */
  it('should subscribe to market data', async () => {
    sandbox.stub(client, 'subscribeToMarketData').resolves();
    let promise = api.subscribeToMarketData('EURUSD', undefined, 1);
    api.terminalState.onSymbolPricesUpdated('1:ps-mpa-1', [{ time: new Date(), symbol: 'EURUSD', bid: 1, ask: 1.1 }]);
    await promise;
    _sinon2.default.assert.calledWith(client.subscribeToMarketData, 'accountId', 1, 'EURUSD', [{ type: 'quotes' }]);
    _sinon2.default.assert.match(api.subscriptions('EURUSD'), [{ type: 'quotes' }]);
    await api.subscribeToMarketData('EURUSD', [{ type: 'books' }, { type: 'candles', timeframe: '1m' }], 1);
    _sinon2.default.assert.match(api.subscriptions('EURUSD'), [{ type: 'quotes' }, { type: 'books' }, { type: 'candles', timeframe: '1m' }]);
    await api.subscribeToMarketData('EURUSD', [{ type: 'quotes' }, { type: 'candles', timeframe: '5m' }], 1);
    _sinon2.default.assert.match(api.subscriptions('EURUSD'), [{ type: 'quotes' }, { type: 'books' }, { type: 'candles', timeframe: '1m' }, { type: 'candles', timeframe: '5m' }]);
  });

  /**
   * @test {MetaApiConnection#subscribeToMarketData}
   */
  it('should not subscribe to symbol that has no specification', async () => {
    sandbox.stub(client, 'subscribeToMarketData').resolves();
    api.terminalState.onSymbolPricesUpdated('1:ps-mpa-1', [{ time: new Date(), symbol: 'EURUSD', bid: 1, ask: 1.1 }]);
    api.terminalState.onSymbolPricesUpdated('1:ps-mpa-1', [{ time: new Date(), symbol: 'AAAAA', bid: 1, ask: 1.1 }]);
    try {
      await api.subscribeToMarketData('AAAAA');
      throw new Error('ValidationError extected');
    } catch (err) {
      err.name.should.equal('ValidationError');
    }
  });

  /**
   * @test {MetaApiConnection#unsubscribeFromMarketData}
   */
  it('should unsubscribe from market data', async () => {
    await api.terminalState.onSymbolPricesUpdated('1:ps-mpa-1', [{ time: new Date(), symbol: 'EURUSD', bid: 1, ask: 1.1 }]);
    sandbox.stub(client, 'unsubscribeFromMarketData').resolves();
    await api.unsubscribeFromMarketData('EURUSD', [{ type: 'quotes' }], 1);
    _sinon2.default.assert.calledWith(client.unsubscribeFromMarketData, 'accountId', 1, 'EURUSD', [{ type: 'quotes' }]);
    await api.subscribeToMarketData('EURUSD', [{ type: 'quotes' }, { type: 'books' }, { type: 'candles', timeframe: '1m' }, { type: 'candles', timeframe: '5m' }], 1);
    _sinon2.default.assert.match(api.subscriptions('EURUSD'), [{ type: 'quotes' }, { type: 'books' }, { type: 'candles', timeframe: '1m' }, { type: 'candles', timeframe: '5m' }]);
    await api.unsubscribeFromMarketData('EURUSD', [{ type: 'quotes' }, { type: 'candles', timeframe: '5m' }], 1);
    _sinon2.default.assert.match(api.subscriptions('EURUSD'), [{ type: 'books' }, { type: 'candles', timeframe: '1m' }]);
  });

  describe('onSubscriptionDowngrade', () => {

    /**
     * @test {MetaApiConnection#onSubscriptionDowngrade}
     */
    it('should unsubscribe during market data subscription downgrade', async () => {
      sandbox.stub(api, 'subscribeToMarketData').resolves();
      sandbox.stub(api, 'unsubscribeFromMarketData').resolves();
      await api.onSubscriptionDowngraded('1:ps-mpa-1', 'EURUSD', undefined, [{ type: 'ticks' }, { type: 'books' }]);
      _sinon2.default.assert.calledWith(api.unsubscribeFromMarketData, 'EURUSD', [{ type: 'ticks' }, { type: 'books' }]);
      _sinon2.default.assert.notCalled(api.subscribeToMarketData);
    });

    /**
     * @test {MetaApiConnection#onSubscriptionDowngrade}
     */
    it('should update market data subscription on downgrade', async () => {
      sandbox.stub(api, 'subscribeToMarketData').resolves();
      sandbox.stub(api, 'unsubscribeFromMarketData').resolves();
      await api.onSubscriptionDowngraded('1:ps-mpa-1', 'EURUSD', [{ type: 'quotes', intervalInMilliseconds: 30000 }]);
      _sinon2.default.assert.calledWith(api.subscribeToMarketData, 'EURUSD', [{ type: 'quotes', intervalInMilliseconds: 30000 }]);
      _sinon2.default.assert.notCalled(api.unsubscribeFromMarketData);
    });
  });

  /**
   * @test {MetaApiConnection#saveUptime}
   */
  it('should save uptime stats to the server', async () => {
    sandbox.stub(client, 'saveUptime').resolves();
    await api.saveUptime({ '1h': 100 });
    _sinon2.default.assert.calledWith(client.saveUptime, 'accountId', { '1h': 100 });
  });

  /**
   * @test {MetaApiConnection#terminalState}
   * @test {MetaApiConnection#historyStorage}
   */
  it('should initialize listeners, terminal state and history storage for accounts with user synch mode', async () => {
    sandbox.stub(client, 'addSynchronizationListener').returns();
    api = new _streamingMetaApiConnection2.default(client, { id: 'accountId' }, undefined, connectionRegistry);
    _should2.default.exist(api.terminalState);
    _should2.default.exist(api.historyStorage);
    _sinon2.default.assert.calledWith(client.addSynchronizationListener, 'accountId', api);
    _sinon2.default.assert.calledWith(client.addSynchronizationListener, 'accountId', api.terminalState);
    _sinon2.default.assert.calledWith(client.addSynchronizationListener, 'accountId', api.historyStorage);
  });

  /**
   * @test {MetaApiConnection#addSynchronizationListener}
   */
  it('should add synchronization listeners', async () => {
    sandbox.stub(client, 'addSynchronizationListener').returns();
    api = new _streamingMetaApiConnection2.default(client, { id: 'accountId' }, undefined, connectionRegistry);
    let listener = {};
    api.addSynchronizationListener(listener);
    _sinon2.default.assert.calledWith(client.addSynchronizationListener, 'accountId', listener);
  });

  /**
   * @test {MetaApiConnection#removeSynchronizationListener}
   */
  it('should remove synchronization listeners', async () => {
    sandbox.stub(client, 'removeSynchronizationListener').returns();
    api = new _streamingMetaApiConnection2.default(client, { id: 'accountId' }, undefined, connectionRegistry);
    let listener = {};
    api.removeSynchronizationListener(listener);
    _sinon2.default.assert.calledWith(client.removeSynchronizationListener, 'accountId', listener);
  });

  /**
   * @test {MetaApiConnection#onConnected}
   */
  it('should sychronize on connection', async () => {
    sandbox.stub(client, 'synchronize').resolves();
    sandbox.stub(_randomstring2.default, 'generate').returns('synchronizationId');
    api = new _streamingMetaApiConnection2.default(client, { id: 'accountId' }, undefined, connectionRegistry);
    api.historyStorage.onHistoryOrderAdded('1:ps-mpa-1', { doneTime: new Date('2020-01-01T00:00:00.000Z') });
    api.historyStorage.onDealAdded('1:ps-mpa-1', { time: new Date('2020-01-02T00:00:00.000Z') });
    await api.onConnected('1:ps-mpa-1', 1);
    await new _promise2.default(res => setTimeout(res, 50));
    _sinon2.default.assert.calledWith(client.synchronize, 'accountId', 1, 'ps-mpa-1', 'synchronizationId', new Date('2020-01-01T00:00:00.000Z'), new Date('2020-01-02T00:00:00.000Z'), null, null, null);
  });

  /**
   * @test {MetaApiConnection#onConnected}
   */
  it('should maintain synchronization if connection has failed', async () => {
    let stub = sandbox.stub(client, 'synchronize');
    stub.onFirstCall().throws(new Error('test error'));
    stub.onSecondCall().resolves();
    sandbox.stub(_randomstring2.default, 'generate').returns('synchronizationId');
    api = new _streamingMetaApiConnection2.default(client, { id: 'accountId' }, undefined, connectionRegistry);
    await api.historyStorage.onHistoryOrderAdded('1:ps-mpa-1', { doneTime: new Date('2020-01-01T00:00:00.000Z') });
    await api.historyStorage.onDealAdded('1:ps-mpa-1', { time: new Date('2020-01-02T00:00:00.000Z') });
    await api.onConnected('1:ps-mpa-1', 1);
    await new _promise2.default(res => setTimeout(res, 50));
    _sinon2.default.assert.calledWith(client.synchronize, 'accountId', 1, 'ps-mpa-1', 'synchronizationId', new Date('2020-01-01T00:00:00.000Z'), new Date('2020-01-02T00:00:00.000Z'), null, null, null);
  });

  /**
   * @test {MetaApiConnection#onConnected}
   */
  it('should not synchronize if connection is closed', async () => {
    let synchronizeStub = sandbox.stub(client, 'synchronize');
    api = new _streamingMetaApiConnection2.default(client, { id: 'accountId' }, undefined, connectionRegistry);
    await api.historyStorage.onHistoryOrderAdded('1:ps-mpa-1', { doneTime: new Date('2020-01-01T00:00:00.000Z') });
    await api.historyStorage.onDealAdded('1:ps-mpa-1', { time: new Date('2020-01-02T00:00:00.000Z') });
    await api.close();
    await api.onConnected('1:ps-mpa-1', 1);
    _sinon2.default.assert.notCalled(synchronizeStub);
  });

  /**
   * @test {MetaApiConnection#close}
   */
  it('should unsubscribe from events on close', async () => {
    sandbox.stub(client, 'addSynchronizationListener').returns();
    sandbox.stub(client, 'removeSynchronizationListener').returns();
    sandbox.stub(client, 'unsubscribe').resolves();
    sandbox.stub(connectionRegistry, 'remove').returns();
    api = new _streamingMetaApiConnection2.default(client, { id: 'accountId' }, undefined, connectionRegistry);
    await api.close();
    _sinon2.default.assert.calledWith(client.unsubscribe, 'accountId');
    _sinon2.default.assert.calledWith(client.removeSynchronizationListener, 'accountId', api);
    _sinon2.default.assert.calledWith(client.removeSynchronizationListener, 'accountId', api.terminalState);
    _sinon2.default.assert.calledWith(client.removeSynchronizationListener, 'accountId', api.historyStorage);
    _sinon2.default.assert.calledWith(connectionRegistry.remove, 'accountId');
  });

  describe('waitSynchronized', () => {

    /**
     * @test {MetaApiConnection#waitSynchronized}
     */
    it('should wait util synchronization complete', async () => {
      sandbox.stub(client, 'waitSynchronized').resolves();
      _sinon2.default.assert.match((await api.isSynchronized('1:ps-mpa-1')), false);
      (await api.isSynchronized()).should.equal(false);
      let promise = api.waitSynchronized({ applicationPattern: 'app.*', synchronizationId: 'synchronizationId',
        timeoutInSeconds: 1, intervalInMilliseconds: 10 });
      let startTime = Date.now();
      await _promise2.default.race([promise, new _promise2.default(res => setTimeout(res, 50))]);
      (Date.now() - startTime).should.be.approximately(50, 10);
      api.onHistoryOrdersSynchronized('1:ps-mpa-1', 'synchronizationId');
      api.onDealsSynchronized('1:ps-mpa-1', 'synchronizationId');
      startTime = Date.now();
      await promise;
      (Date.now() - startTime).should.be.approximately(10, 10);
      (await api.isSynchronized('1:ps-mpa-1', 'synchronizationId')).should.equal(true);
    });

    /**
     * @test {MetaApiConnection#waitSynchronized}
     */
    it('should time out waiting for synchronization complete', async () => {
      try {
        await api.waitSynchronized({ applicationPattern: 'app.*', synchronizationId: 'synchronizationId',
          timeoutInSeconds: 1, intervalInMilliseconds: 10 });
        throw new Error('TimeoutError is expected');
      } catch (err) {
        err.name.should.equal('TimeoutError');
      }
      (await api.isSynchronized('synchronizationId')).should.equal(false);
    });
  });

  /**
   * @test {MetaApiConnection#initialize}
   */
  it('should load data to history storage from disk', async () => {
    sandbox.stub(api.historyStorage, 'initialize').resolves();
    await api.initialize();
    _sinon2.default.assert.calledOnce(api.historyStorage.initialize);
  });

  /**
   * @test {MetaApiConnection#onDisconnected}
   */
  it('should set synchronized false on disconnect', async () => {
    await api.onConnected('1:ps-mpa-1', 2);
    await new _promise2.default(res => setTimeout(res, 50));
    _sinon2.default.assert.match(api.synchronized, true);
    await api.onDisconnected('1:ps-mpa-1');
    _sinon2.default.assert.match(api.synchronized, false);
  });

  /**
   * @test {MetaApiConnection#onDisconnected}
   */
  it('should delete state if stream closed', async () => {
    await api.onConnected('1:ps-mpa-1', 2);
    await new _promise2.default(res => setTimeout(res, 50));
    _sinon2.default.assert.match(api.synchronized, true);
    await api.onStreamClosed('1:ps-mpa-1');
    _sinon2.default.assert.match(api.synchronized, false);
  });

  /**
   * @test {MetaApiConnection#onDisconnected}
   */
  it('should create refresh subscriptions job', async () => {
    sandbox.stub(client, 'refreshMarketDataSubscriptions').resolves();
    sandbox.stub(client, 'subscribeToMarketData').resolves();
    sandbox.stub(client, 'waitSynchronized').resolves();
    await api.onSynchronizationStarted('1:ps-mpa-1');
    await clock.tickAsync(50);
    _sinon2.default.assert.calledWith(client.refreshMarketDataSubscriptions, 'accountId', 1, []);
    api.terminalState.onSymbolPricesUpdated('1:ps-mpa-1', [{ time: new Date(), symbol: 'EURUSD', bid: 1, ask: 1.1 }]);
    await api.subscribeToMarketData('EURUSD', [{ type: 'quotes' }], 1);
    await clock.tickAsync(1050);
    _sinon2.default.assert.calledWith(client.refreshMarketDataSubscriptions, 'accountId', 1, [{ symbol: 'EURUSD', subscriptions: [{ type: 'quotes' }] }]);
    _sinon2.default.assert.callCount(client.refreshMarketDataSubscriptions, 2);
    await api.onDisconnected('1:ps-mpa-1');
    await clock.tickAsync(1050);
    _sinon2.default.assert.callCount(client.refreshMarketDataSubscriptions, 2);
    await api.onSynchronizationStarted('1:ps-mpa-1');
    await clock.tickAsync(50);
    _sinon2.default.assert.callCount(client.refreshMarketDataSubscriptions, 3);
    await api.close();
    await clock.tickAsync(1050);
    _sinon2.default.assert.callCount(client.refreshMarketDataSubscriptions, 3);
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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