'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _keys = require('babel-runtime/core-js/object/keys');

var _keys2 = _interopRequireDefault(_keys);

var _entries = require('babel-runtime/core-js/object/entries');

var _entries2 = _interopRequireDefault(_entries);

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _stringify = require('babel-runtime/core-js/json/stringify');

var _stringify2 = _interopRequireDefault(_stringify);

var _values = require('babel-runtime/core-js/object/values');

var _values2 = _interopRequireDefault(_values);

var _cryptoJs = require('crypto-js');

var _cryptoJs2 = _interopRequireDefault(_cryptoJs);

var _synchronizationListener = require('../clients/metaApi/synchronizationListener');

var _synchronizationListener2 = _interopRequireDefault(_synchronizationListener);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Responsible for storing a local copy of remote terminal state
 */
class TerminalState extends _synchronizationListener2.default {

  /**
   * Constructs the instance of terminal state class
   */
  constructor() {
    super();
    this._stateByInstanceIndex = {};
    this._waitForPriceResolves = {};
    this._combinedState = {
      accountInformation: undefined,
      positions: [],
      orders: [],
      specificationsBySymbol: {},
      pricesBySymbol: {},
      completedOrders: {},
      removedPositions: {},
      ordersInitialized: false,
      positionsInitialized: false,
      lastUpdateTime: 0
    };
  }

  /**
   * Returns true if MetaApi have connected to MetaTrader terminal
   * @return {Boolean} true if MetaApi have connected to MetaTrader terminal
   */
  get connected() {
    return (0, _values2.default)(this._stateByInstanceIndex).reduce((acc, s) => acc || s.connected, false);
  }

  /**
   * Returns true if MetaApi have connected to MetaTrader terminal and MetaTrader terminal is connected to broker
   * @return {Boolean} true if MetaApi have connected to MetaTrader terminal and MetaTrader terminal is connected to
   * broker
   */
  get connectedToBroker() {
    return (0, _values2.default)(this._stateByInstanceIndex).reduce((acc, s) => acc || s.connectedToBroker, false);
  }

  /**
   * Returns a local copy of account information
   * @returns {MetatraderAccountInformation} local copy of account information
   */
  get accountInformation() {
    return this._combinedState.accountInformation;
  }

  /**
   * Returns a local copy of MetaTrader positions opened
   * @returns {Array<MetatraderPosition>} a local copy of MetaTrader positions opened
   */
  get positions() {
    return this._combinedState.positions;
  }

  /**
   * Returns a local copy of MetaTrader orders opened
   * @returns {Array<MetatraderOrder>} a local copy of MetaTrader orders opened
   */
  get orders() {
    return this._combinedState.orders;
  }

  /**
   * Returns a local copy of symbol specifications available in MetaTrader trading terminal
   * @returns {Array<MetatraderSymbolSpecification>} a local copy of symbol specifications available in MetaTrader
   * trading terminal
   */
  get specifications() {
    return (0, _values2.default)(this._combinedState.specificationsBySymbol);
  }

  /**
   * Returns hashes of terminal state data for incremental synchronization
   * @param {String} accountType account type
   * @param {String} instanceIndex index of instance to get hashes of
   * @returns {Object} hashes of terminal state data
   */
  getHashes(accountType, instanceIndex) {
    const state = this._getState(instanceIndex);

    const sortByKey = (obj1, obj2, key) => {
      if (obj1[key] < obj2[key]) {
        return -1;
      }
      if (obj1[key] > obj2[key]) {
        return 1;
      }
      return 0;
    };
    const specifications = JSON.parse((0, _stringify2.default)((0, _values2.default)(state.specificationsBySymbol)));
    specifications.sort((a, b) => sortByKey(a, b, 'symbol'));
    if (accountType === 'cloud-g1') {
      specifications.forEach(specification => {
        delete specification.description;
      });
    }
    const specificationsHash = specifications.length ? this._getHash(specifications, accountType, ['digits']) : null;

    const positions = JSON.parse((0, _stringify2.default)(state.positions));
    positions.sort((a, b) => sortByKey(a, b, 'id'));
    positions.forEach(position => {
      delete position.profit;
      delete position.unrealizedProfit;
      delete position.realizedProfit;
      delete position.currentPrice;
      delete position.currentTickValue;
      delete position.updateSequenceNumber;
      delete position.accountCurrencyExchangeRate;
      delete position.comment;
      delete position.brokerComment;
      delete position.clientId;
      if (accountType === 'cloud-g1') {
        delete position.time;
        delete position.updateTime;
      }
    });
    const positionsHash = state.positionsInitialized ? this._getHash(positions, accountType, ['magic']) : null;

    const orders = JSON.parse((0, _stringify2.default)(state.orders));
    orders.sort((a, b) => sortByKey(a, b, 'id'));
    orders.forEach(order => {
      delete order.currentPrice;
      delete order.updateSequenceNumber;
      delete order.accountCurrencyExchangeRate;
      delete order.comment;
      delete order.brokerComment;
      delete order.clientId;
      if (accountType === 'cloud-g1') {
        delete order.time;
      }
    });
    const ordersHash = state.ordersInitialized ? this._getHash(orders, accountType, ['magic']) : null;

    return {
      specificationsMd5: specificationsHash,
      positionsMd5: positionsHash,
      ordersMd5: ordersHash
    };
  }

  /**
   * Returns MetaTrader symbol specification by symbol
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @return {MetatraderSymbolSpecification} MetatraderSymbolSpecification found or undefined if specification for a
   * symbol is not found
   */
  specification(symbol) {
    return this._combinedState.specificationsBySymbol[symbol];
  }

  /**
   * Returns MetaTrader symbol price by symbol
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @return {MetatraderSymbolPrice} MetatraderSymbolPrice found or undefined if price for a symbol is not found
   */
  price(symbol) {
    return this._combinedState.pricesBySymbol[symbol];
  }

  /**
   * Waits for price to be received
   * @param {string} symbol symbol (e.g. currency pair or an index)
   * @param {number} [timeoutInSeconds] timeout in seconds, default is 30
   * @return {Promise<MetatraderSymbolPrice>} promise resolving with price or undefined if price has not been received
   */
  async waitForPrice(symbol, timeoutInSeconds = 30) {
    this._waitForPriceResolves[symbol] = this._waitForPriceResolves[symbol] || [];
    if (!this.price(symbol)) {
      await _promise2.default.race([new _promise2.default(res => this._waitForPriceResolves[symbol].push(res)), new _promise2.default(res => setTimeout(res, timeoutInSeconds * 1000))]);
    }
    return this.price(symbol);
  }

  /**
   * Invoked when connection to MetaTrader terminal established
   * @param {String} instanceIndex index of an account instance connected
   */
  onConnected(instanceIndex) {
    this._getState(instanceIndex).connected = true;
  }

  /**
   * Invoked when connection to MetaTrader terminal terminated
   * @param {String} instanceIndex index of an account instance connected
   */
  onDisconnected(instanceIndex) {
    let state = this._getState(instanceIndex);
    state.connected = false;
    state.connectedToBroker = false;
  }

  /**
   * Invoked when broker connection status have changed
   * @param {String} instanceIndex index of an account instance connected
   * @param {Boolean} connected is MetaTrader terminal is connected to broker
   */
  onBrokerConnectionStatusChanged(instanceIndex, connected) {
    this._getState(instanceIndex).connectedToBroker = connected;
  }

  /**
   * Invoked when MetaTrader terminal state synchronization is started
   * @param {String} instanceIndex index of an account instance connected
   * @param {Boolean} specificationsUpdated whether specifications are going to be updated during synchronization
   * @param {Boolean} positionsUpdated whether positions are going to be updated during synchronization
   * @param {Boolean} ordersUpdated whether orders are going to be updated during synchronization
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onSynchronizationStarted(instanceIndex, specificationsUpdated, positionsUpdated, ordersUpdated) {
    let state = this._getState(instanceIndex);
    state.accountInformation = undefined;
    state.pricesBySymbol = {};
    if (positionsUpdated) {
      state.positions = [];
      state.removedPositions = {};
      state.positionsInitialized = false;
    }
    if (ordersUpdated) {
      state.orders = [];
      state.completedOrders = {};
      state.ordersInitialized = false;
    }
    if (specificationsUpdated) {
      state.specificationsBySymbol = {};
    }
  }

  /**
   * Invoked when MetaTrader account information is updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderAccountInformation} accountInformation updated MetaTrader account information
   */
  onAccountInformationUpdated(instanceIndex, accountInformation) {
    let state = this._getState(instanceIndex);
    state.accountInformation = accountInformation;
    this._combinedState.accountInformation = accountInformation;
  }

  /**
   * Invoked when the positions are replaced as a result of initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {Array<MetatraderPosition>} positions updated array of positions
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onPositionsReplaced(instanceIndex, positions) {
    let state = this._getState(instanceIndex);
    state.positions = positions;
  }

  /**
   * Invoked when position synchronization fnished to indicate progress of an initial terminal state synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onPositionsSynchronized(instanceIndex, synchronizationId) {
    let state = this._getState(instanceIndex);
    state.removedPositions = {};
    state.positionsInitialized = true;
  }

  /**
   * Invoked when MetaTrader position is updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderPosition} position updated MetaTrader position
   */
  onPositionUpdated(instanceIndex, position) {
    let instanceState = this._getState(instanceIndex);

    const updatePosition = state => {
      let index = state.positions.findIndex(p => p.id === position.id);
      if (index !== -1) {
        state.positions[index] = position;
      } else if (!state.removedPositions[position.id]) {
        state.positions.push(position);
      }
    };
    updatePosition(instanceState);
    updatePosition(this._combinedState);
  }

  /**
   * Invoked when MetaTrader position is removed
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} positionId removed MetaTrader position id
   */
  onPositionRemoved(instanceIndex, positionId) {
    let instanceState = this._getState(instanceIndex);

    const removePosition = state => {
      let position = state.positions.find(p => p.id === positionId);
      if (!position) {
        for (let e of (0, _entries2.default)(state.removedPositions)) {
          if (e[1] + 5 * 60 * 1000 < Date.now()) {
            delete state.removedPositions[e[0]];
          }
        }
        state.removedPositions[positionId] = Date.now();
      } else {
        state.positions = state.positions.filter(p => p.id !== positionId);
      }
    };
    removePosition(instanceState);
    removePosition(this._combinedState);
  }

  /**
   * Invoked when the orders are replaced as a result of initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {Array<MetatraderOrder>} orders updated array of pending orders
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onPendingOrdersReplaced(instanceIndex, orders) {
    let state = this._getState(instanceIndex);
    state.orders = orders;
  }

  /**
   * Invoked when pending order synchronization fnished to indicate progress of an initial terminal state
   * synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPendingOrdersSynchronized(instanceIndex, synchronizationId) {
    let state = this._getState(instanceIndex);
    state.completedOrders = {};
    state.positionsInitialized = true;
    state.ordersInitialized = true;
    this._combinedState.accountInformation = state.accountInformation;
    this._combinedState.positions = state.positions;
    this._combinedState.orders = state.orders;
    this._combinedState.specificationsBySymbol = state.specificationsBySymbol;
    this._combinedState.positionsInitialized = true;
    this._combinedState.ordersInitialized = true;
    this._combinedState.completedOrders = {};
    this._combinedState.removedPositions = {};
  }

  /**
   * Invoked when MetaTrader pending order is updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderOrder} order updated MetaTrader pending order
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onPendingOrderUpdated(instanceIndex, order) {
    let instanceState = this._getState(instanceIndex);

    const updatePendingOrder = state => {
      let index = state.orders.findIndex(o => o.id === order.id);
      if (index !== -1) {
        state.orders[index] = order;
      } else if (!state.completedOrders[order.id]) {
        state.orders.push(order);
      }
    };
    updatePendingOrder(instanceState);
    updatePendingOrder(this._combinedState);
  }

  /**
   * Invoked when MetaTrader pending order is completed (executed or canceled)
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} orderId completed MetaTrader pending order id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onPendingOrderCompleted(instanceIndex, orderId) {
    let instanceState = this._getState(instanceIndex);

    const completeOrder = state => {
      let order = state.orders.find(o => o.id === orderId);
      if (!order) {
        for (let e of (0, _entries2.default)(state.completedOrders)) {
          if (e[1] + 5 * 60 * 1000 < Date.now()) {
            delete state.completedOrders[e[0]];
          }
        }
        state.completedOrders[orderId] = Date.now();
      } else {
        state.orders = state.orders.filter(o => o.id !== orderId);
      }
    };
    completeOrder(instanceState);
    completeOrder(this._combinedState);
  }

  /**
   * Invoked when a symbol specification was updated
   * @param {String} instanceIndex index of account instance connected
   * @param {Array<MetatraderSymbolSpecification>} specifications updated specifications
   * @param {Array<String>} removedSymbols removed symbols
   */
  onSymbolSpecificationsUpdated(instanceIndex, specifications, removedSymbols) {
    let instanceState = this._getState(instanceIndex);

    const updateSpecifications = state => {
      for (let specification of specifications) {
        state.specificationsBySymbol[specification.symbol] = specification;
      }
      for (let symbol of removedSymbols) {
        delete state.specificationsBySymbol[symbol];
      }
    };
    updateSpecifications(instanceState);
    updateSpecifications(this._combinedState);
  }

  /**
   * Invoked when prices for several symbols were updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {Array<MetatraderSymbolPrice>} prices updated MetaTrader symbol prices
   * @param {Number} equity account liquidation value
   * @param {Number} margin margin used
   * @param {Number} freeMargin free margin
   * @param {Number} marginLevel margin level calculated as % of equity/margin
   */
  // eslint-disable-next-line complexity
  onSymbolPricesUpdated(instanceIndex, prices, equity, margin, freeMargin, marginLevel) {
    let instanceState = this._getState(instanceIndex);

    // eslint-disable-next-line complexity
    const updateSymbolPrices = state => {
      state.lastUpdateTime = Math.max(prices.map(p => p.time.getTime()));
      let pricesInitialized = false;
      for (let price of prices || []) {
        state.pricesBySymbol[price.symbol] = price;
        let positions = state.positions.filter(p => p.symbol === price.symbol);
        let otherPositions = state.positions.filter(p => p.symbol !== price.symbol);
        let orders = state.orders.filter(o => o.symbol === price.symbol);
        pricesInitialized = true;
        for (let position of otherPositions) {
          let p = state.pricesBySymbol[position.symbol];
          if (p) {
            if (position.unrealizedProfit === undefined) {
              this._updatePositionProfits(position, p);
            }
          } else {
            pricesInitialized = false;
          }
        }
        for (let position of positions) {
          this._updatePositionProfits(position, price);
        }
        for (let order of orders) {
          order.currentPrice = order.type === 'ORDER_TYPE_BUY' || order.type === 'ORDER_TYPE_BUY_LIMIT' || order.type === 'ORDER_TYPE_BUY_STOP' || order.type === 'ORDER_TYPE_BUY_STOP_LIMIT' ? price.ask : price.bid;
        }
        let priceResolves = this._waitForPriceResolves[price.symbol] || [];
        if (priceResolves.length) {
          for (let resolve of priceResolves) {
            resolve();
          }
          delete this._waitForPriceResolves[price.symbol];
        }
      }
      if (state.accountInformation) {
        if (state.positionsInitialized && pricesInitialized) {
          if (state.accountInformation.platform === 'mt5') {
            state.accountInformation.equity = equity !== undefined ? equity : state.accountInformation.balance + state.positions.reduce((acc, p) => acc + Math.round((p.unrealizedProfit || 0) * 100) / 100 + Math.round((p.swap || 0) * 100) / 100, 0);
          } else {
            state.accountInformation.equity = equity !== undefined ? equity : state.accountInformation.balance + state.positions.reduce((acc, p) => acc + Math.round((p.swap || 0) * 100) / 100 + Math.round((p.commission || 0) * 100) / 100 + Math.round((p.unrealizedProfit || 0) * 100) / 100, 0);
          }
          state.accountInformation.equity = Math.round(state.accountInformation.equity * 100) / 100;
        } else {
          state.accountInformation.equity = equity !== undefined ? equity : state.accountInformation.equity;
        }
        state.accountInformation.margin = margin !== undefined ? margin : state.accountInformation.margin;
        state.accountInformation.freeMargin = freeMargin !== undefined ? freeMargin : state.accountInformation.freeMargin;
        state.accountInformation.marginLevel = freeMargin !== undefined ? marginLevel : state.accountInformation.marginLevel;
      }
    };
    updateSymbolPrices(instanceState);
    updateSymbolPrices(this._combinedState);
  }

  /**
   * Invoked when a stream for an instance index is closed
   * @param {String} instanceIndex index of an account instance connected
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onStreamClosed(instanceIndex) {
    delete this._stateByInstanceIndex[instanceIndex];
  }

  // eslint-disable-next-line complexity
  _updatePositionProfits(position, price) {
    let specification = this.specification(position.symbol);
    if (specification) {
      let multiplier = Math.pow(10, specification.digits);
      if (position.profit !== undefined) {
        position.profit = Math.round(position.profit * multiplier) / multiplier;
      }
      if (position.unrealizedProfit === undefined || position.realizedProfit === undefined) {
        position.unrealizedProfit = (position.type === 'POSITION_TYPE_BUY' ? 1 : -1) * (position.currentPrice - position.openPrice) * position.currentTickValue * position.volume / specification.tickSize;
        position.unrealizedProfit = Math.round(position.unrealizedProfit * multiplier) / multiplier;
        position.realizedProfit = position.profit - position.unrealizedProfit;
      }
      let newPositionPrice = position.type === 'POSITION_TYPE_BUY' ? price.bid : price.ask;
      let isProfitable = (position.type === 'POSITION_TYPE_BUY' ? 1 : -1) * (newPositionPrice - position.openPrice);
      let currentTickValue = isProfitable > 0 ? price.profitTickValue : price.lossTickValue;
      let unrealizedProfit = (position.type === 'POSITION_TYPE_BUY' ? 1 : -1) * (newPositionPrice - position.openPrice) * currentTickValue * position.volume / specification.tickSize;
      unrealizedProfit = Math.round(unrealizedProfit * multiplier) / multiplier;
      position.unrealizedProfit = unrealizedProfit;
      position.profit = position.unrealizedProfit + position.realizedProfit;
      position.profit = Math.round(position.profit * multiplier) / multiplier;
      position.currentPrice = newPositionPrice;
      position.currentTickValue = currentTickValue;
    }
  }

  _getState(instanceIndex) {
    if (!this._stateByInstanceIndex['' + instanceIndex]) {
      this._stateByInstanceIndex['' + instanceIndex] = this._constructTerminalState(instanceIndex);
    }
    return this._stateByInstanceIndex['' + instanceIndex];
  }

  _constructTerminalState(instanceIndex) {
    return {
      instanceIndex,
      connected: false,
      connectedToBroker: false,
      accountInformation: undefined,
      positions: [],
      orders: [],
      specificationsBySymbol: {},
      pricesBySymbol: {},
      completedOrders: {},
      removedPositions: {},
      ordersInitialized: false,
      positionsInitialized: false,
      lastUpdateTime: 0
    };
  }

  _getHash(obj, accountType, integerKeys) {
    let jsonItem = '';
    if (accountType === 'cloud-g1') {
      const stringify = (objFromJson, key) => {
        if (typeof objFromJson === 'number') {
          if (integerKeys.includes(key)) {
            return objFromJson;
          } else {
            return objFromJson.toFixed(8);
          }
        } else if (Array.isArray(objFromJson)) {
          return `[${objFromJson.map(item => stringify(item)).join(',')}]`;
        } else if (objFromJson === null) {
          return objFromJson;
        } else if (typeof objFromJson !== 'object' || objFromJson.getTime) {
          return (0, _stringify2.default)(objFromJson);
        }

        let props = (0, _keys2.default)(objFromJson).map(keyItem => `"${keyItem}":${stringify(objFromJson[keyItem], keyItem)}`).join(',');
        return `{${props}}`;
      };

      jsonItem = stringify(obj);
    } else if (accountType === 'cloud-g2') {
      jsonItem = (0, _stringify2.default)(obj);
    }
    return _cryptoJs2.default.MD5(jsonItem).toString();
  }

}
exports.default = TerminalState;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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